// uniprops.cpp
// copyright c 2003 Raphael Finkel.
// Some code taken from Bram Moolenaar's Vim.
// license: Gnu Public License.

// routines to manipulate Unicode characters based on their properties:
// 	isCombining: tells if a character is a combining character (often accents)
// 	isAlphabetic: tells if a character is alphabetic (includes combining)
// 	unPrecompose: converts a Unicode string to an equivalent one, expanding
// 		precomposed characters such as ñ to components such as ñ
// 	reduce: converts a Unicode string to a simplified one, removing all
// 		combining characters, expanding precomposed forms, and transcribing,
// 		which introduces "sounds-like" substitutions.
// 	toUpper: converts a Unicode string to an upper-case equivalent
//
//      isRightDirection: tells if a character reads from right to left
//
// 	These methods use sorted tables and binary search.
//

#include <stdlib.h>
#include <string.h>

#include "myparameters.h"
#include "uniprops.h"

// combining, precompose tables and routines

// the following struct and tables automatically generated by combineExtract.pl
// from UnicodeData-4.0.0

typedef struct
{
	wide_t rangeStart;
	wide_t rangeEnd;
	wide_t replacement;
} tableEntry;

tableEntry combineTable[] = { // Unicode 4.0.0
	{0x300,0x357}, {0x35d,0x36f}, {0x483,0x486}, {0x488,0x489}, 
	{0x591,0x5a1}, {0x5a3,0x5b9}, {0x5bb,0x5bd}, {0x5bf,0x5bf}, 
	{0x5c1,0x5c2}, {0x5c4,0x5c4}, {0x610,0x615}, {0x64b,0x658}, 
	{0x670,0x670}, {0x6d6,0x6dc}, {0x6de,0x6e4}, {0x6e7,0x6e8}, 
	{0x6ea,0x6ed}, {0x711,0x711}, {0x730,0x74a}, {0x7a6,0x7b0}, 
	{0x901,0x903}, {0x93c,0x93c}, {0x93e,0x94d}, {0x951,0x954}, 
	{0x962,0x963}, {0x981,0x983}, {0x9bc,0x9bc}, {0x9be,0x9c4}, 
	{0x9c7,0x9c8}, {0x9cb,0x9cd}, {0x9d7,0x9d7}, {0x9e2,0x9e3}, 
	{0xa01,0xa03}, {0xa3c,0xa3c}, {0xa3e,0xa42}, {0xa47,0xa48}, 
	{0xa4b,0xa4d}, {0xa70,0xa71}, {0xa81,0xa83}, {0xabc,0xabc}, 
	{0xabe,0xac5}, {0xac7,0xac9}, {0xacb,0xacd}, {0xae2,0xae3}, 
	{0xb01,0xb03}, {0xb3c,0xb3c}, {0xb3e,0xb43}, {0xb47,0xb48}, 
	{0xb4b,0xb4d}, {0xb56,0xb57}, {0xb82,0xb82}, {0xbbe,0xbc2}, 
	{0xbc6,0xbc8}, {0xbca,0xbcd}, {0xbd7,0xbd7}, {0xc01,0xc03}, 
	{0xc3e,0xc44}, {0xc46,0xc48}, {0xc4a,0xc4d}, {0xc55,0xc56}, 
	{0xc82,0xc83}, {0xcbc,0xcbc}, {0xcbe,0xcc4}, {0xcc6,0xcc8}, 
	{0xcca,0xccd}, {0xcd5,0xcd6}, {0xd02,0xd03}, {0xd3e,0xd43}, 
	{0xd46,0xd48}, {0xd4a,0xd4d}, {0xd57,0xd57}, {0xd82,0xd83}, 
	{0xdca,0xdca}, {0xdcf,0xdd4}, {0xdd6,0xdd6}, {0xdd8,0xddf}, 
	{0xdf2,0xdf3}, {0xe31,0xe31}, {0xe34,0xe3a}, {0xe47,0xe4e}, 
	{0xeb1,0xeb1}, {0xeb4,0xeb9}, {0xebb,0xebc}, {0xec8,0xecd}, 
	{0xf18,0xf19}, {0xf35,0xf35}, {0xf37,0xf37}, {0xf39,0xf39}, 
	{0xf3e,0xf3f}, {0xf71,0xf84}, {0xf86,0xf87}, {0xf90,0xf97}, 
	{0xf99,0xfbc}, {0xfc6,0xfc6}, {0x102c,0x1032}, {0x1036,0x1039}, 
	{0x1056,0x1059}, {0x1712,0x1714}, {0x1732,0x1734}, 
	{0x1752,0x1753}, {0x1772,0x1773}, {0x17b6,0x17d3}, 
	{0x17dd,0x17dd}, {0x180b,0x180d}, {0x18a9,0x18a9}, 
	{0x1920,0x192b}, {0x1930,0x193b}, {0x20d0,0x20ea}, 
	{0x302a,0x302f}, {0x3099,0x309a}, {0xfb1e,0xfb1e}, 
	{0xfe00,0xfe0f}, {0xfe20,0xfe23}, {0x1d165,0x1d169}, 
	{0x1d16d,0x1d172}, {0x1d17b,0x1d182}, {0x1d185,0x1d18b}, 
	{0x1d1aa,0x1d1ad}, {0xe0100,0xe01ef}, 
}; // combineTable

tableEntry alphabeticTable[] = { // Unicode 4.0.0
	// uniset +cat=Lm +cat=Lo +cat=Lu +cat=Ll +cat=Lt +cat=Mn +cat=Me +cat=Mc c
  { 0x0041, 0x005A }, { 0x0061, 0x007A }, { 0x00AA, 0x00AA },
  { 0x00B5, 0x00B5 }, { 0x00BA, 0x00BA }, { 0x00C0, 0x00D6 },
  { 0x00D8, 0x00F6 }, { 0x00F8, 0x0236 }, { 0x0250, 0x02C1 },
  { 0x02C6, 0x02D1 }, { 0x02E0, 0x02E4 }, { 0x02EE, 0x02EE },
  { 0x0300, 0x0357 }, { 0x035D, 0x036F }, { 0x037A, 0x037A },
  { 0x0386, 0x0386 }, { 0x0388, 0x038A }, { 0x038C, 0x038C },
  { 0x038E, 0x03A1 }, { 0x03A3, 0x03CE }, { 0x03D0, 0x03F5 },
  { 0x03F7, 0x03FB }, { 0x0400, 0x0481 }, { 0x0483, 0x0486 },
  { 0x0488, 0x04CE }, { 0x04D0, 0x04F5 }, { 0x04F8, 0x04F9 },
  { 0x0500, 0x050F }, { 0x0531, 0x0556 }, { 0x0559, 0x0559 },
  { 0x0561, 0x0587 }, { 0x0591, 0x05A1 }, { 0x05A3, 0x05B9 },
  { 0x05BB, 0x05BD }, { 0x05BF, 0x05BF }, { 0x05C1, 0x05C2 },
  { 0x05C4, 0x05C4 }, { 0x05D0, 0x05EA }, { 0x05F0, 0x05F2 },
  { 0x0610, 0x0615 }, { 0x0621, 0x063A }, { 0x0640, 0x0658 },
  { 0x066E, 0x06D3 }, { 0x06D5, 0x06DC }, { 0x06DE, 0x06E8 },
  { 0x06EA, 0x06EF }, { 0x06FA, 0x06FC }, { 0x06FF, 0x06FF },
  { 0x0710, 0x074A }, { 0x074D, 0x074F }, { 0x0780, 0x07B1 },
  { 0x0901, 0x0939 }, { 0x093C, 0x094D }, { 0x0950, 0x0954 },
  { 0x0958, 0x0963 }, { 0x0981, 0x0983 }, { 0x0985, 0x098C },
  { 0x098F, 0x0990 }, { 0x0993, 0x09A8 }, { 0x09AA, 0x09B0 },
  { 0x09B2, 0x09B2 }, { 0x09B6, 0x09B9 }, { 0x09BC, 0x09C4 },
  { 0x09C7, 0x09C8 }, { 0x09CB, 0x09CD }, { 0x09D7, 0x09D7 },
  { 0x09DC, 0x09DD }, { 0x09DF, 0x09E3 }, { 0x09F0, 0x09F1 },
  { 0x0A01, 0x0A03 }, { 0x0A05, 0x0A0A }, { 0x0A0F, 0x0A10 },
  { 0x0A13, 0x0A28 }, { 0x0A2A, 0x0A30 }, { 0x0A32, 0x0A33 },
  { 0x0A35, 0x0A36 }, { 0x0A38, 0x0A39 }, { 0x0A3C, 0x0A3C },
  { 0x0A3E, 0x0A42 }, { 0x0A47, 0x0A48 }, { 0x0A4B, 0x0A4D },
  { 0x0A59, 0x0A5C }, { 0x0A5E, 0x0A5E }, { 0x0A70, 0x0A74 },
  { 0x0A81, 0x0A83 }, { 0x0A85, 0x0A8D }, { 0x0A8F, 0x0A91 },
  { 0x0A93, 0x0AA8 }, { 0x0AAA, 0x0AB0 }, { 0x0AB2, 0x0AB3 },
  { 0x0AB5, 0x0AB9 }, { 0x0ABC, 0x0AC5 }, { 0x0AC7, 0x0AC9 },
  { 0x0ACB, 0x0ACD }, { 0x0AD0, 0x0AD0 }, { 0x0AE0, 0x0AE3 },
  { 0x0B01, 0x0B03 }, { 0x0B05, 0x0B0C }, { 0x0B0F, 0x0B10 },
  { 0x0B13, 0x0B28 }, { 0x0B2A, 0x0B30 }, { 0x0B32, 0x0B33 },
  { 0x0B35, 0x0B39 }, { 0x0B3C, 0x0B43 }, { 0x0B47, 0x0B48 },
  { 0x0B4B, 0x0B4D }, { 0x0B56, 0x0B57 }, { 0x0B5C, 0x0B5D },
  { 0x0B5F, 0x0B61 }, { 0x0B71, 0x0B71 }, { 0x0B82, 0x0B83 },
  { 0x0B85, 0x0B8A }, { 0x0B8E, 0x0B90 }, { 0x0B92, 0x0B95 },
  { 0x0B99, 0x0B9A }, { 0x0B9C, 0x0B9C }, { 0x0B9E, 0x0B9F },
  { 0x0BA3, 0x0BA4 }, { 0x0BA8, 0x0BAA }, { 0x0BAE, 0x0BB5 },
  { 0x0BB7, 0x0BB9 }, { 0x0BBE, 0x0BC2 }, { 0x0BC6, 0x0BC8 },
  { 0x0BCA, 0x0BCD }, { 0x0BD7, 0x0BD7 }, { 0x0C01, 0x0C03 },
  { 0x0C05, 0x0C0C }, { 0x0C0E, 0x0C10 }, { 0x0C12, 0x0C28 },
  { 0x0C2A, 0x0C33 }, { 0x0C35, 0x0C39 }, { 0x0C3E, 0x0C44 },
  { 0x0C46, 0x0C48 }, { 0x0C4A, 0x0C4D }, { 0x0C55, 0x0C56 },
  { 0x0C60, 0x0C61 }, { 0x0C82, 0x0C83 }, { 0x0C85, 0x0C8C },
  { 0x0C8E, 0x0C90 }, { 0x0C92, 0x0CA8 }, { 0x0CAA, 0x0CB3 },
  { 0x0CB5, 0x0CB9 }, { 0x0CBC, 0x0CC4 }, { 0x0CC6, 0x0CC8 },
  { 0x0CCA, 0x0CCD }, { 0x0CD5, 0x0CD6 }, { 0x0CDE, 0x0CDE },
  { 0x0CE0, 0x0CE1 }, { 0x0D02, 0x0D03 }, { 0x0D05, 0x0D0C },
  { 0x0D0E, 0x0D10 }, { 0x0D12, 0x0D28 }, { 0x0D2A, 0x0D39 },
  { 0x0D3E, 0x0D43 }, { 0x0D46, 0x0D48 }, { 0x0D4A, 0x0D4D },
  { 0x0D57, 0x0D57 }, { 0x0D60, 0x0D61 }, { 0x0D82, 0x0D83 },
  { 0x0D85, 0x0D96 }, { 0x0D9A, 0x0DB1 }, { 0x0DB3, 0x0DBB },
  { 0x0DBD, 0x0DBD }, { 0x0DC0, 0x0DC6 }, { 0x0DCA, 0x0DCA },
  { 0x0DCF, 0x0DD4 }, { 0x0DD6, 0x0DD6 }, { 0x0DD8, 0x0DDF },
  { 0x0DF2, 0x0DF3 }, { 0x0E01, 0x0E3A }, { 0x0E40, 0x0E4E },
  { 0x0E81, 0x0E82 }, { 0x0E84, 0x0E84 }, { 0x0E87, 0x0E88 },
  { 0x0E8A, 0x0E8A }, { 0x0E8D, 0x0E8D }, { 0x0E94, 0x0E97 },
  { 0x0E99, 0x0E9F }, { 0x0EA1, 0x0EA3 }, { 0x0EA5, 0x0EA5 },
  { 0x0EA7, 0x0EA7 }, { 0x0EAA, 0x0EAB }, { 0x0EAD, 0x0EB9 },
  { 0x0EBB, 0x0EBD }, { 0x0EC0, 0x0EC4 }, { 0x0EC6, 0x0EC6 },
  { 0x0EC8, 0x0ECD }, { 0x0EDC, 0x0EDD }, { 0x0F00, 0x0F00 },
  { 0x0F18, 0x0F19 }, { 0x0F35, 0x0F35 }, { 0x0F37, 0x0F37 },
  { 0x0F39, 0x0F39 }, { 0x0F3E, 0x0F47 }, { 0x0F49, 0x0F6A },
  { 0x0F71, 0x0F84 }, { 0x0F86, 0x0F8B }, { 0x0F90, 0x0F97 },
  { 0x0F99, 0x0FBC }, { 0x0FC6, 0x0FC6 }, { 0x1000, 0x1021 },
  { 0x1023, 0x1027 }, { 0x1029, 0x102A }, { 0x102C, 0x1032 },
  { 0x1036, 0x1039 }, { 0x1050, 0x1059 }, { 0x10A0, 0x10C5 },
  { 0x10D0, 0x10F8 }, { 0x1100, 0x1159 }, { 0x115F, 0x11A2 },
  { 0x11A8, 0x11F9 }, { 0x1200, 0x1206 }, { 0x1208, 0x1246 },
  { 0x1248, 0x1248 }, { 0x124A, 0x124D }, { 0x1250, 0x1256 },
  { 0x1258, 0x1258 }, { 0x125A, 0x125D }, { 0x1260, 0x1286 },
  { 0x1288, 0x1288 }, { 0x128A, 0x128D }, { 0x1290, 0x12AE },
  { 0x12B0, 0x12B0 }, { 0x12B2, 0x12B5 }, { 0x12B8, 0x12BE },
  { 0x12C0, 0x12C0 }, { 0x12C2, 0x12C5 }, { 0x12C8, 0x12CE },
  { 0x12D0, 0x12D6 }, { 0x12D8, 0x12EE }, { 0x12F0, 0x130E },
  { 0x1310, 0x1310 }, { 0x1312, 0x1315 }, { 0x1318, 0x131E },
  { 0x1320, 0x1346 }, { 0x1348, 0x135A }, { 0x13A0, 0x13F4 },
  { 0x1401, 0x166C }, { 0x166F, 0x1676 }, { 0x1681, 0x169A },
  { 0x16A0, 0x16EA }, { 0x1700, 0x170C }, { 0x170E, 0x1714 },
  { 0x1720, 0x1734 }, { 0x1740, 0x1753 }, { 0x1760, 0x176C },
  { 0x176E, 0x1770 }, { 0x1772, 0x1773 }, { 0x1780, 0x17B3 },
  { 0x17B6, 0x17D3 }, { 0x17D7, 0x17D7 }, { 0x17DC, 0x17DD },
  { 0x180B, 0x180D }, { 0x1820, 0x1877 }, { 0x1880, 0x18A9 },
  { 0x1900, 0x191C }, { 0x1920, 0x192B }, { 0x1930, 0x193B },
  { 0x1950, 0x196D }, { 0x1970, 0x1974 }, { 0x1D00, 0x1D6B },
  { 0x1E00, 0x1E9B }, { 0x1EA0, 0x1EF9 }, { 0x1F00, 0x1F15 },
  { 0x1F18, 0x1F1D }, { 0x1F20, 0x1F45 }, { 0x1F48, 0x1F4D },
  { 0x1F50, 0x1F57 }, { 0x1F59, 0x1F59 }, { 0x1F5B, 0x1F5B },
  { 0x1F5D, 0x1F5D }, { 0x1F5F, 0x1F7D }, { 0x1F80, 0x1FB4 },
  { 0x1FB6, 0x1FBC }, { 0x1FBE, 0x1FBE }, { 0x1FC2, 0x1FC4 },
  { 0x1FC6, 0x1FCC }, { 0x1FD0, 0x1FD3 }, { 0x1FD6, 0x1FDB },
  { 0x1FE0, 0x1FEC }, { 0x1FF2, 0x1FF4 }, { 0x1FF6, 0x1FFC },
  { 0x2071, 0x2071 }, { 0x207F, 0x207F }, { 0x20D0, 0x20EA },
  { 0x2102, 0x2102 }, { 0x2107, 0x2107 }, { 0x210A, 0x2113 },
  { 0x2115, 0x2115 }, { 0x2119, 0x211D }, { 0x2124, 0x2124 },
  { 0x2126, 0x2126 }, { 0x2128, 0x2128 }, { 0x212A, 0x212D },
  { 0x212F, 0x2131 }, { 0x2133, 0x2139 }, { 0x213D, 0x213F },
  { 0x2145, 0x2149 }, { 0x3005, 0x3006 }, { 0x302A, 0x302F },
  { 0x3031, 0x3035 }, { 0x303B, 0x303C }, { 0x3041, 0x3096 },
  { 0x3099, 0x309A }, { 0x309D, 0x309F }, { 0x30A1, 0x30FA },
  { 0x30FC, 0x30FF }, { 0x3105, 0x312C }, { 0x3131, 0x318E },
  { 0x31A0, 0x31B7 }, { 0x31F0, 0x31FF }, { 0xA000, 0xA48C },
  { 0xF900, 0xFA2D }, { 0xFA30, 0xFA6A }, { 0xFB00, 0xFB06 },
  { 0xFB13, 0xFB17 }, { 0xFB1D, 0xFB28 }, { 0xFB2A, 0xFB36 },
  { 0xFB38, 0xFB3C }, { 0xFB3E, 0xFB3E }, { 0xFB40, 0xFB41 },
  { 0xFB43, 0xFB44 }, { 0xFB46, 0xFBB1 }, { 0xFBD3, 0xFD3D },
  { 0xFD50, 0xFD8F }, { 0xFD92, 0xFDC7 }, { 0xFDF0, 0xFDFB },
  { 0xFE00, 0xFE0F }, { 0xFE20, 0xFE23 }, { 0xFE70, 0xFE74 },
  { 0xFE76, 0xFEFC }, { 0xFF21, 0xFF3A }, { 0xFF41, 0xFF5A },
  { 0xFF66, 0xFFBE }, { 0xFFC2, 0xFFC7 }, { 0xFFCA, 0xFFCF },
  { 0xFFD2, 0xFFD7 }, { 0xFFDA, 0xFFDC }, { 0x10000, 0x1000B },
  { 0x1000D, 0x10026 }, { 0x10028, 0x1003A }, { 0x1003C, 0x1003D },
  { 0x1003F, 0x1004D }, { 0x10050, 0x1005D }, { 0x10080, 0x100FA },
  { 0x10300, 0x1031E }, { 0x10330, 0x10349 }, { 0x10380, 0x1039D },
  { 0x10400, 0x1049D }, { 0x10800, 0x10805 }, { 0x10808, 0x10808 },
  { 0x1080A, 0x10835 }, { 0x10837, 0x10838 }, { 0x1083C, 0x1083C },
  { 0x1083F, 0x1083F }, { 0x1D165, 0x1D169 }, { 0x1D16D, 0x1D172 },
  { 0x1D17B, 0x1D182 }, { 0x1D185, 0x1D18B }, { 0x1D1AA, 0x1D1AD },
  { 0x1D400, 0x1D454 }, { 0x1D456, 0x1D49C }, { 0x1D49E, 0x1D49F },
  { 0x1D4A2, 0x1D4A2 }, { 0x1D4A5, 0x1D4A6 }, { 0x1D4A9, 0x1D4AC },
  { 0x1D4AE, 0x1D4B9 }, { 0x1D4BB, 0x1D4BB }, { 0x1D4BD, 0x1D4C3 },
  { 0x1D4C5, 0x1D505 }, { 0x1D507, 0x1D50A }, { 0x1D50D, 0x1D514 },
  { 0x1D516, 0x1D51C }, { 0x1D51E, 0x1D539 }, { 0x1D53B, 0x1D53E },
  { 0x1D540, 0x1D544 }, { 0x1D546, 0x1D546 }, { 0x1D54A, 0x1D550 },
  { 0x1D552, 0x1D6A3 }, { 0x1D6A8, 0x1D6C0 }, { 0x1D6C2, 0x1D6DA },
  { 0x1D6DC, 0x1D6FA }, { 0x1D6FC, 0x1D714 }, { 0x1D716, 0x1D734 },
  { 0x1D736, 0x1D74E }, { 0x1D750, 0x1D76E }, { 0x1D770, 0x1D788 },
  { 0x1D78A, 0x1D7A8 }, { 0x1D7AA, 0x1D7C2 }, { 0x1D7C4, 0x1D7C9 },
  { 0x2F800, 0x2FA1D }, { 0xE0100, 0xE01EF }
}; // alphabeticTable

typedef struct
{
        int rangeStart;
        int rangeEnd;
} punctuationStruct;

punctuationStruct punctuationTable[] = {
        {0x5be,0x5be}, {0x5c0,0x5c0}, {0x5c3,0x5c3}, {0x5d0,0x5ea},
        {0x5f0,0x5f4}, {0x200f,0x200f}, {0x202a,0x202b}, {0x202d,0x202e},
        {0xfb1d,0xfb1d}, {0xfb1f,0xfb28}, {0xfb2a,0xfb36},
        {0xfb38,0xfb3c}, {0xfb3e,0xfb3e}, {0xfb40,0xfb41},
        {0xfb43,0xfb44}, {0xfb46,0xfb4f}, {0x10800,0x10805},
        {0x10808,0x10808}, {0x1080a,0x10835}, {0x10837,0x10838},
        {0x1083c,0x1083c}, {0x1083f,0x1083f},
        };

static int inpunctuationStruct(punctuationStruct *table, int length, wide_t c) {
    int mid, bot, top;
    /* binary search in table */
    bot = 0;
    top = length - 1;
    while (top >= bot) {
		mid = (bot + top) / 2;
		if (table[mid].rangeEnd < c)
			bot = mid + 1;
		else if (table[mid].rangeStart > c)
			top = mid - 1;
		else
			return mid;
	}
	return -1;
} // inpunctuationStruct

// returns a bool: 1 if c reads from right to left, 0 if not.
int isRightDirection(wide_t c) {
	if (inpunctuationStruct(punctuationTable, 
            sizeof(punctuationTable) / sizeof(punctuationStruct), c) == -1)
		return(0);
	else
		return(1);
} // isRightDirection

// the following code taken from Vim 6.1 source (mbyte.c); this code originally
// written by Raphael Finkel

// Return index in table if "c" is in "table[size]", else -1.  
static int inTable(tableEntry *table, int length, wide_t c) {
    int mid, bot, top;
    /* first quick check for Latin1 etc. characters */
    if (c < table[0].rangeStart) return -1;
    /* binary search in table */
    bot = 0;
    top = length - 1;
    while (top >= bot) {
		mid = (bot + top) / 2;
		if (table[mid].rangeEnd < c)
			bot = mid + 1;
		else if (table[mid].rangeStart > c)
			top = mid - 1;
		else
			return mid;
	}
	return -1;
} // inTable

// returns a bool: 1 if c is a combining character, 0 if not.
int isCombining(wide_t c) {
	if (inTable(combineTable, sizeof(combineTable) / sizeof(tableEntry), c) ==
			-1)
		return(0);
	else
		return(1);
} // isCombining

// returns a bool: 1 if c is an alphabetic character, 0 if not.
int isAlphabetic(wide_t c) {
	if (inTable(alphabeticTable, sizeof(alphabeticTable) / sizeof(tableEntry),
			c) == -1)
		return(0);
	else
		return(1);
} // isAlphabetic

// the following code taken from Vim 6.1 source (mbyte.c); this code originally
// written by Raphael Finkel

typedef struct
{
    int rangeStart;
    int rangeEnd;
    int step;
    int offset;
} convertStruct;

static int wide_convert(int a, convertStruct table[], int tableSize) {
/*
 * Generic conversion function for case operations.
 * Return the converted equivalent of "a", which is a UCS-4 character.  Use
 * the given conversion "table".  Uses binary search on "table".
 */
    int start, mid, end; /* indices into table */
    start = 0;
    end = tableSize / sizeof(convertStruct);
    while (start < end) {
		/* need to search further */
		mid = (end + start) /2;
		if (table[mid].rangeEnd < a)
			start = mid + 1;
		else
			end = mid;
	}
	if (table[start].rangeStart <= a && a <= table[start].rangeEnd
			&& (a - table[start].rangeStart) % table[start].step == 0)
		return (a + table[start].offset);
	else
		return a;
} // wide_convert

/*
 * The following table built by upperLowerExtract.pl < UnicodeData.txt .
 * They must be in numeric order, because we use binary search on them.
 * An entry such as {0x41,0x5a,1,32} means that UCS-4 characters in the range
 * from 0x41 to 0x5a inclusive, stepping by 1, are switched to lower (for
 * example) by adding 32.
 */
convertStruct toLowerTable[] = {
	{0x41,0x5a,1,32}, {0xc0,0xd6,1,32}, {0xd8,0xde,1,32}, 
	{0x100,0x12e,2,1}, {0x130,0x130,-1,-199}, {0x132,0x136,2,1}, 
	{0x139,0x147,2,1}, {0x14a,0x176,2,1}, {0x178,0x178,-1,-121}, 
	{0x179,0x17d,2,1}, {0x181,0x181,-1,210}, {0x182,0x184,2,1}, 
	{0x186,0x186,-1,206}, {0x187,0x187,-1,1}, {0x189,0x18a,1,205}, 
	{0x18b,0x18b,-1,1}, {0x18e,0x18e,-1,79}, {0x18f,0x18f,-1,202}, 
	{0x190,0x190,-1,203}, {0x191,0x191,-1,1}, {0x193,0x193,-1,205}, 
	{0x194,0x194,-1,207}, {0x196,0x196,-1,211}, {0x197,0x197,-1,209}, 
	{0x198,0x198,-1,1}, {0x19c,0x19c,-1,211}, {0x19d,0x19d,-1,213}, 
	{0x19f,0x19f,-1,214}, {0x1a0,0x1a4,2,1}, {0x1a6,0x1a6,-1,218}, 
	{0x1a7,0x1a7,-1,1}, {0x1a9,0x1a9,-1,218}, {0x1ac,0x1ac,-1,1}, 
	{0x1ae,0x1ae,-1,218}, {0x1af,0x1af,-1,1}, {0x1b1,0x1b2,1,217}, 
	{0x1b3,0x1b5,2,1}, {0x1b7,0x1b7,-1,219}, {0x1b8,0x1bc,4,1}, 
	{0x1c4,0x1ca,3,2}, {0x1cd,0x1db,2,1}, {0x1de,0x1ee,2,1}, 
	{0x1f1,0x1f1,-1,2}, {0x1f4,0x1f4,-1,1}, {0x1f6,0x1f6,-1,-97}, 
	{0x1f7,0x1f7,-1,-56}, {0x1f8,0x21e,2,1}, {0x220,0x220,-1,-130}, 
	{0x222,0x232,2,1}, {0x386,0x386,-1,38}, {0x388,0x38a,1,37}, 
	{0x38c,0x38c,-1,64}, {0x38e,0x38f,1,63}, {0x391,0x3a1,1,32}, 
	{0x3a3,0x3ab,1,32}, {0x3d8,0x3ee,2,1}, {0x3f4,0x3f4,-1,-60}, 
	{0x3f7,0x3f7,-1,1}, {0x3f9,0x3f9,-1,-7}, {0x3fa,0x3fa,-1,1}, 
	{0x400,0x40f,1,80}, {0x410,0x42f,1,32}, {0x460,0x480,2,1}, 
	{0x48a,0x4be,2,1}, {0x4c1,0x4cd,2,1}, {0x4d0,0x4f4,2,1}, 
	{0x4f8,0x500,8,1}, {0x502,0x50e,2,1}, {0x531,0x556,1,48}, 
	{0x1e00,0x1e94,2,1}, {0x1ea0,0x1ef8,2,1}, {0x1f08,0x1f0f,1,-8}, 
	{0x1f18,0x1f1d,1,-8}, {0x1f28,0x1f2f,1,-8}, {0x1f38,0x1f3f,1,-8}, 
	{0x1f48,0x1f4d,1,-8}, {0x1f59,0x1f5f,2,-8}, {0x1f68,0x1f6f,1,-8}, 
	{0x1fb8,0x1fb9,1,-8}, {0x1fba,0x1fbb,1,-74}, {0x1fc8,0x1fcb,1,-86}, 
	{0x1fd8,0x1fd9,1,-8}, {0x1fda,0x1fdb,1,-100}, {0x1fe8,0x1fe9,1,-8}, 
	{0x1fea,0x1feb,1,-112}, {0x1fec,0x1fec,-1,-7}, {0x1ff8,0x1ff9,1,-128}, 
	{0x1ffa,0x1ffb,1,-126}, {0x2126,0x2126,-1,-7517}, {0x212a,0x212a,-1,-8383}, 
	{0x212b,0x212b,-1,-8262}, {0xff21,0xff3a,1,32}, {0x10400,0x10427,1,40} 
}; // toLowerTable

convertStruct toUpperTable[] = {
	{0x61,0x7a,1,-32}, {0xb5,0xb5,-1,743}, {0xe0,0xf6,1,-32}, 
	{0xf8,0xfe,1,-32}, {0xff,0xff,-1,121}, {0x101,0x12f,2,-1}, 
	{0x131,0x131,-1,-232}, {0x133,0x137,2,-1}, {0x13a,0x148,2,-1}, 
	{0x14b,0x177,2,-1}, {0x17a,0x17e,2,-1}, {0x17f,0x17f,-1,-300}, 
	{0x183,0x185,2,-1}, {0x188,0x18c,4,-1}, {0x192,0x192,-1,-1}, 
	{0x195,0x195,-1,97}, {0x199,0x199,-1,-1}, {0x19e,0x19e,-1,130}, 
	{0x1a1,0x1a5,2,-1}, {0x1a8,0x1ad,5,-1}, {0x1b0,0x1b4,4,-1}, 
	{0x1b6,0x1b9,3,-1}, {0x1bd,0x1bd,-1,-1}, {0x1bf,0x1bf,-1,56}, 
	{0x1c5,0x1c6,1,-1}, {0x1c8,0x1c9,1,-1}, {0x1cb,0x1cc,1,-1}, 
	{0x1ce,0x1dc,2,-1}, {0x1dd,0x1dd,-1,-79}, {0x1df,0x1ef,2,-1}, 
	{0x1f2,0x1f3,1,-1}, {0x1f5,0x1f9,4,-1}, {0x1fb,0x21f,2,-1}, 
	{0x223,0x233,2,-1}, {0x253,0x253,-1,-210}, {0x254,0x254,-1,-206}, 
	{0x256,0x257,1,-205}, {0x259,0x259,-1,-202}, {0x25b,0x25b,-1,-203}, 
	{0x260,0x260,-1,-205}, {0x263,0x263,-1,-207}, {0x268,0x268,-1,-209}, 
	{0x269,0x26f,6,-211}, {0x272,0x272,-1,-213}, {0x275,0x275,-1,-214}, 
	{0x280,0x283,3,-218}, {0x288,0x288,-1,-218}, {0x28a,0x28b,1,-217}, 
	{0x292,0x292,-1,-219}, {0x3ac,0x3ac,-1,-38}, {0x3ad,0x3af,1,-37}, 
	{0x3b1,0x3c1,1,-32}, {0x3c2,0x3c2,-1,-31}, {0x3c3,0x3cb,1,-32}, 
	{0x3cc,0x3cc,-1,-64}, {0x3cd,0x3ce,1,-63}, {0x3d0,0x3d0,-1,-62}, 
	{0x3d1,0x3d1,-1,-57}, {0x3d5,0x3d5,-1,-47}, {0x3d6,0x3d6,-1,-54}, 
	{0x3d9,0x3ef,2,-1}, {0x3f0,0x3f0,-1,-86}, {0x3f1,0x3f1,-1,-80}, 
	{0x3f2,0x3f2,-1,7}, {0x3f5,0x3f5,-1,-96}, {0x3f8,0x3fb,3,-1}, 
	{0x430,0x44f,1,-32}, {0x450,0x45f,1,-80}, {0x461,0x481,2,-1}, 
	{0x48b,0x4bf,2,-1}, {0x4c2,0x4ce,2,-1}, {0x4d1,0x4f5,2,-1}, 
	{0x4f9,0x501,8,-1}, {0x503,0x50f,2,-1}, {0x561,0x586,1,-48}, 
	{0x1e01,0x1e95,2,-1}, {0x1e9b,0x1e9b,-1,-59}, {0x1ea1,0x1ef9,2,-1}, 
	{0x1f00,0x1f07,1,8}, {0x1f10,0x1f15,1,8}, {0x1f20,0x1f27,1,8}, 
	{0x1f30,0x1f37,1,8}, {0x1f40,0x1f45,1,8}, {0x1f51,0x1f57,2,8}, 
	{0x1f60,0x1f67,1,8}, {0x1f70,0x1f71,1,74}, {0x1f72,0x1f75,1,86}, 
	{0x1f76,0x1f77,1,100}, {0x1f78,0x1f79,1,128}, {0x1f7a,0x1f7b,1,112}, 
	{0x1f7c,0x1f7d,1,126}, {0x1f80,0x1f87,1,8}, {0x1f90,0x1f97,1,8}, 
	{0x1fa0,0x1fa7,1,8}, {0x1fb0,0x1fb1,1,8}, {0x1fb3,0x1fb3,-1,9}, 
	{0x1fbe,0x1fbe,-1,-7205}, {0x1fc3,0x1fc3,-1,9}, {0x1fd0,0x1fd1,1,8}, 
	{0x1fe0,0x1fe1,1,8}, {0x1fe5,0x1fe5,-1,7}, {0x1ff3,0x1ff3,-1,9}, 
	{0xff41,0xff5a,1,-32}, {0x10428,0x1044f,1,-40} 
}; // toUpperTable

void toUpper(wide_t *dest, const wide_t *source, int sourceLength) {
	for (; sourceLength; sourceLength -= 1) { // one wide_t
		*dest++ = wide_convert(*source++, toUpperTable, sizeof(toUpperTable));
	} // one wide_t
} // toUpper

typedef struct
{
    wide_t ucsChar;
    wide_t first;
    wide_t second;
} precomposeStruct;

precomposeStruct precomposeTable[] =
{
	{0x00C0, 0x0041, 0x0300}, {0x00C1, 0x0041, 0x0301}, 
	{0x00C2, 0x0041, 0x0302}, {0x00C3, 0x0041, 0x0303}, 
	{0x00C4, 0x0041, 0x0308}, {0x00C5, 0x0041, 0x030A}, 
	{0x00C7, 0x0043, 0x0327}, {0x00C8, 0x0045, 0x0300}, 
	{0x00C9, 0x0045, 0x0301}, {0x00CA, 0x0045, 0x0302}, 
	{0x00CB, 0x0045, 0x0308}, {0x00CC, 0x0049, 0x0300}, 
	{0x00CD, 0x0049, 0x0301}, {0x00CE, 0x0049, 0x0302}, 
	{0x00CF, 0x0049, 0x0308}, {0x00D1, 0x004E, 0x0303}, 
	{0x00D2, 0x004F, 0x0300}, {0x00D3, 0x004F, 0x0301}, 
	{0x00D4, 0x004F, 0x0302}, {0x00D5, 0x004F, 0x0303}, 
	{0x00D6, 0x004F, 0x0308}, {0x00D9, 0x0055, 0x0300}, 
	{0x00DA, 0x0055, 0x0301}, {0x00DB, 0x0055, 0x0302}, 
	{0x00DC, 0x0055, 0x0308}, {0x00DD, 0x0059, 0x0301}, 
	{0x00E0, 0x0061, 0x0300}, {0x00E1, 0x0061, 0x0301}, 
	{0x00E2, 0x0061, 0x0302}, {0x00E3, 0x0061, 0x0303}, 
	{0x00E4, 0x0061, 0x0308}, {0x00E5, 0x0061, 0x030A}, 
	{0x00E7, 0x0063, 0x0327}, {0x00E8, 0x0065, 0x0300}, 
	{0x00E9, 0x0065, 0x0301}, {0x00EA, 0x0065, 0x0302}, 
	{0x00EB, 0x0065, 0x0308}, {0x00EC, 0x0069, 0x0300}, 
	{0x00ED, 0x0069, 0x0301}, {0x00EE, 0x0069, 0x0302}, 
	{0x00EF, 0x0069, 0x0308}, {0x00F1, 0x006E, 0x0303}, 
	{0x00F2, 0x006F, 0x0300}, {0x00F3, 0x006F, 0x0301}, 
	{0x00F4, 0x006F, 0x0302}, {0x00F5, 0x006F, 0x0303}, 
	{0x00F6, 0x006F, 0x0308}, {0x00F9, 0x0075, 0x0300}, 
	{0x00FA, 0x0075, 0x0301}, {0x00FB, 0x0075, 0x0302}, 
	{0x00FC, 0x0075, 0x0308}, {0x00FD, 0x0079, 0x0301}, 
	{0x00FF, 0x0079, 0x0308}, {0x0100, 0x0041, 0x0304}, 
	{0x0101, 0x0061, 0x0304}, {0x0102, 0x0041, 0x0306}, 
	{0x0103, 0x0061, 0x0306}, {0x0104, 0x0041, 0x0328}, 
	{0x0105, 0x0061, 0x0328}, {0x0106, 0x0043, 0x0301}, 
	{0x0107, 0x0063, 0x0301}, {0x0108, 0x0043, 0x0302}, 
	{0x0109, 0x0063, 0x0302}, {0x010A, 0x0043, 0x0307}, 
	{0x010B, 0x0063, 0x0307}, {0x010C, 0x0043, 0x030C}, 
	{0x010D, 0x0063, 0x030C}, {0x010E, 0x0044, 0x030C}, 
	{0x010F, 0x0064, 0x030C}, {0x0112, 0x0045, 0x0304}, 
	{0x0113, 0x0065, 0x0304}, {0x0114, 0x0045, 0x0306}, 
	{0x0115, 0x0065, 0x0306}, {0x0116, 0x0045, 0x0307}, 
	{0x0117, 0x0065, 0x0307}, {0x0118, 0x0045, 0x0328}, 
	{0x0119, 0x0065, 0x0328}, {0x011A, 0x0045, 0x030C}, 
	{0x011B, 0x0065, 0x030C}, {0x011C, 0x0047, 0x0302}, 
	{0x011D, 0x0067, 0x0302}, {0x011E, 0x0047, 0x0306}, 
	{0x011F, 0x0067, 0x0306}, {0x0120, 0x0047, 0x0307}, 
	{0x0121, 0x0067, 0x0307}, {0x0122, 0x0047, 0x0327}, 
	{0x0123, 0x0067, 0x0327}, {0x0124, 0x0048, 0x0302}, 
	{0x0125, 0x0068, 0x0302}, {0x0128, 0x0049, 0x0303}, 
	{0x0129, 0x0069, 0x0303}, {0x012A, 0x0049, 0x0304}, 
	{0x012B, 0x0069, 0x0304}, {0x012C, 0x0049, 0x0306}, 
	{0x012D, 0x0069, 0x0306}, {0x012E, 0x0049, 0x0328}, 
	{0x012F, 0x0069, 0x0328}, {0x0130, 0x0049, 0x0307}, 
	{0x0134, 0x004A, 0x0302}, {0x0135, 0x006A, 0x0302}, 
	{0x0136, 0x004B, 0x0327}, {0x0137, 0x006B, 0x0327}, 
	{0x0139, 0x004C, 0x0301}, {0x013A, 0x006C, 0x0301}, 
	{0x013B, 0x004C, 0x0327}, {0x013C, 0x006C, 0x0327}, 
	{0x013D, 0x004C, 0x030C}, {0x013E, 0x006C, 0x030C}, 
	{0x0143, 0x004E, 0x0301}, {0x0144, 0x006E, 0x0301}, 
	{0x0145, 0x004E, 0x0327}, {0x0146, 0x006E, 0x0327}, 
	{0x0147, 0x004E, 0x030C}, {0x0148, 0x006E, 0x030C}, 
	{0x014C, 0x004F, 0x0304}, {0x014D, 0x006F, 0x0304}, 
	{0x014E, 0x004F, 0x0306}, {0x014F, 0x006F, 0x0306}, 
	{0x0150, 0x004F, 0x030B}, {0x0151, 0x006F, 0x030B}, 
	{0x0154, 0x0052, 0x0301}, {0x0155, 0x0072, 0x0301}, 
	{0x0156, 0x0052, 0x0327}, {0x0157, 0x0072, 0x0327}, 
	{0x0158, 0x0052, 0x030C}, {0x0159, 0x0072, 0x030C}, 
	{0x015A, 0x0053, 0x0301}, {0x015B, 0x0073, 0x0301}, 
	{0x015C, 0x0053, 0x0302}, {0x015D, 0x0073, 0x0302}, 
	{0x015E, 0x0053, 0x0327}, {0x015F, 0x0073, 0x0327}, 
	{0x0160, 0x0053, 0x030C}, {0x0161, 0x0073, 0x030C}, 
	{0x0162, 0x0054, 0x0327}, {0x0163, 0x0074, 0x0327}, 
	{0x0164, 0x0054, 0x030C}, {0x0165, 0x0074, 0x030C}, 
	{0x0168, 0x0055, 0x0303}, {0x0169, 0x0075, 0x0303}, 
	{0x016A, 0x0055, 0x0304}, {0x016B, 0x0075, 0x0304}, 
	{0x016C, 0x0055, 0x0306}, {0x016D, 0x0075, 0x0306}, 
	{0x016E, 0x0055, 0x030A}, {0x016F, 0x0075, 0x030A}, 
	{0x0170, 0x0055, 0x030B}, {0x0171, 0x0075, 0x030B}, 
	{0x0172, 0x0055, 0x0328}, {0x0173, 0x0075, 0x0328}, 
	{0x0174, 0x0057, 0x0302}, {0x0175, 0x0077, 0x0302}, 
	{0x0176, 0x0059, 0x0302}, {0x0177, 0x0079, 0x0302}, 
	{0x0178, 0x0059, 0x0308}, {0x0179, 0x005A, 0x0301}, 
	{0x017A, 0x007A, 0x0301}, {0x017B, 0x005A, 0x0307}, 
	{0x017C, 0x007A, 0x0307}, {0x017D, 0x005A, 0x030C}, 
	{0x017E, 0x007A, 0x030C}, {0x01A0, 0x004F, 0x031B}, 
	{0x01A1, 0x006F, 0x031B}, {0x01AF, 0x0055, 0x031B}, 
	{0x01B0, 0x0075, 0x031B}, {0x01CD, 0x0041, 0x030C}, 
	{0x01CE, 0x0061, 0x030C}, {0x01CF, 0x0049, 0x030C}, 
	{0x01D0, 0x0069, 0x030C}, {0x01D1, 0x004F, 0x030C}, 
	{0x01D2, 0x006F, 0x030C}, {0x01D3, 0x0055, 0x030C}, 
	{0x01D4, 0x0075, 0x030C}, {0x01D5, 0x00DC, 0x0304}, 
	{0x01D6, 0x00FC, 0x0304}, {0x01D7, 0x00DC, 0x0301}, 
	{0x01D8, 0x00FC, 0x0301}, {0x01D9, 0x00DC, 0x030C}, 
	{0x01DA, 0x00FC, 0x030C}, {0x01DB, 0x00DC, 0x0300}, 
	{0x01DC, 0x00FC, 0x0300}, {0x01DE, 0x00C4, 0x0304}, 
	{0x01DF, 0x00E4, 0x0304}, {0x01E0, 0x0226, 0x0304}, 
	{0x01E1, 0x0227, 0x0304}, {0x01E2, 0x00C6, 0x0304}, 
	{0x01E3, 0x00E6, 0x0304}, {0x01E6, 0x0047, 0x030C}, 
	{0x01E7, 0x0067, 0x030C}, {0x01E8, 0x004B, 0x030C}, 
	{0x01E9, 0x006B, 0x030C}, {0x01EA, 0x004F, 0x0328}, 
	{0x01EB, 0x006F, 0x0328}, {0x01EC, 0x01EA, 0x0304}, 
	{0x01ED, 0x01EB, 0x0304}, {0x01EE, 0x01B7, 0x030C}, 
	{0x01EF, 0x0292, 0x030C}, {0x01F0, 0x006A, 0x030C}, 
	{0x01F4, 0x0047, 0x0301}, {0x01F5, 0x0067, 0x0301}, 
	{0x01F8, 0x004E, 0x0300}, {0x01F9, 0x006E, 0x0300}, 
	{0x01FA, 0x00C5, 0x0301}, {0x01FB, 0x00E5, 0x0301}, 
	{0x01FC, 0x00C6, 0x0301}, {0x01FD, 0x00E6, 0x0301}, 
	{0x01FE, 0x00D8, 0x0301}, {0x01FF, 0x00F8, 0x0301}, 
	{0x0200, 0x0041, 0x030F}, {0x0201, 0x0061, 0x030F}, 
	{0x0202, 0x0041, 0x0311}, {0x0203, 0x0061, 0x0311}, 
	{0x0204, 0x0045, 0x030F}, {0x0205, 0x0065, 0x030F}, 
	{0x0206, 0x0045, 0x0311}, {0x0207, 0x0065, 0x0311}, 
	{0x0208, 0x0049, 0x030F}, {0x0209, 0x0069, 0x030F}, 
	{0x020A, 0x0049, 0x0311}, {0x020B, 0x0069, 0x0311}, 
	{0x020C, 0x004F, 0x030F}, {0x020D, 0x006F, 0x030F}, 
	{0x020E, 0x004F, 0x0311}, {0x020F, 0x006F, 0x0311}, 
	{0x0210, 0x0052, 0x030F}, {0x0211, 0x0072, 0x030F}, 
	{0x0212, 0x0052, 0x0311}, {0x0213, 0x0072, 0x0311}, 
	{0x0214, 0x0055, 0x030F}, {0x0215, 0x0075, 0x030F}, 
	{0x0216, 0x0055, 0x0311}, {0x0217, 0x0075, 0x0311}, 
	{0x0218, 0x0053, 0x0326}, {0x0219, 0x0073, 0x0326}, 
	{0x021A, 0x0054, 0x0326}, {0x021B, 0x0074, 0x0326}, 
	{0x021E, 0x0048, 0x030C}, {0x021F, 0x0068, 0x030C}, 
	{0x0226, 0x0041, 0x0307}, {0x0227, 0x0061, 0x0307}, 
	{0x0228, 0x0045, 0x0327}, {0x0229, 0x0065, 0x0327}, 
	{0x022A, 0x00D6, 0x0304}, {0x022B, 0x00F6, 0x0304}, 
	{0x022C, 0x00D5, 0x0304}, {0x022D, 0x00F5, 0x0304}, 
	{0x022E, 0x004F, 0x0307}, {0x022F, 0x006F, 0x0307}, 
	{0x0230, 0x022E, 0x0304}, {0x0231, 0x022F, 0x0304}, 
	{0x0232, 0x0059, 0x0304}, {0x0233, 0x0079, 0x0304}, 
	{0x0386, 0x0391, 0x0301}, {0x0388, 0x0395, 0x0301}, 
	{0x0389, 0x0397, 0x0301}, {0x038A, 0x0399, 0x0301}, 
	{0x038C, 0x039F, 0x0301}, {0x038E, 0x03A5, 0x0301}, 
	{0x038F, 0x03A9, 0x0301}, {0x0390, 0x03CA, 0x0301}, 
	{0x03AA, 0x0399, 0x0308}, {0x03AB, 0x03A5, 0x0308}, 
	{0x03AC, 0x03B1, 0x0301}, {0x03AD, 0x03B5, 0x0301}, 
	{0x03AE, 0x03B7, 0x0301}, {0x03AF, 0x03B9, 0x0301}, 
	{0x03B0, 0x03CB, 0x0301}, {0x03CA, 0x03B9, 0x0308}, 
	{0x03CB, 0x03C5, 0x0308}, {0x03CC, 0x03BF, 0x0301}, 
	{0x03CD, 0x03C5, 0x0301}, {0x03CE, 0x03C9, 0x0301}, 
	{0x03D3, 0x03D2, 0x0301}, {0x03D4, 0x03D2, 0x0308}, 
	{0x0400, 0x0415, 0x0300}, {0x0401, 0x0415, 0x0308}, 
	{0x0403, 0x0413, 0x0301}, {0x0407, 0x0406, 0x0308}, 
	{0x040C, 0x041A, 0x0301}, {0x040D, 0x0418, 0x0300}, 
	{0x040E, 0x0423, 0x0306}, {0x0419, 0x0418, 0x0306}, 
	{0x0439, 0x0438, 0x0306}, {0x0450, 0x0435, 0x0300}, 
	{0x0451, 0x0435, 0x0308}, {0x0453, 0x0433, 0x0301}, 
	{0x0457, 0x0456, 0x0308}, {0x045C, 0x043A, 0x0301}, 
	{0x045D, 0x0438, 0x0300}, {0x045E, 0x0443, 0x0306}, 
	{0x0476, 0x0474, 0x030F}, {0x0477, 0x0475, 0x030F}, 
	{0x04C1, 0x0416, 0x0306}, {0x04C2, 0x0436, 0x0306}, 
	{0x04D0, 0x0410, 0x0306}, {0x04D1, 0x0430, 0x0306}, 
	{0x04D2, 0x0410, 0x0308}, {0x04D3, 0x0430, 0x0308}, 
	{0x04D6, 0x0415, 0x0306}, {0x04D7, 0x0435, 0x0306}, 
	{0x04DA, 0x04D8, 0x0308}, {0x04DB, 0x04D9, 0x0308}, 
	{0x04DC, 0x0416, 0x0308}, {0x04DD, 0x0436, 0x0308}, 
	{0x04DE, 0x0417, 0x0308}, {0x04DF, 0x0437, 0x0308}, 
	{0x04E2, 0x0418, 0x0304}, {0x04E3, 0x0438, 0x0304}, 
	{0x04E4, 0x0418, 0x0308}, {0x04E5, 0x0438, 0x0308}, 
	{0x04E6, 0x041E, 0x0308}, {0x04E7, 0x043E, 0x0308}, 
	{0x04EA, 0x04E8, 0x0308}, {0x04EB, 0x04E9, 0x0308}, 
	{0x04EC, 0x042D, 0x0308}, {0x04ED, 0x044D, 0x0308}, 
	{0x04EE, 0x0423, 0x0304}, {0x04EF, 0x0443, 0x0304}, 
	{0x04F0, 0x0423, 0x0308}, {0x04F1, 0x0443, 0x0308}, 
	{0x04F2, 0x0423, 0x030B}, {0x04F3, 0x0443, 0x030B}, 
	{0x04F4, 0x0427, 0x0308}, {0x04F5, 0x0447, 0x0308}, 
	{0x04F8, 0x042B, 0x0308}, {0x04F9, 0x044B, 0x0308}, 
	{0x0622, 0x0627, 0x0653}, {0x0623, 0x0627, 0x0654}, 
	{0x0624, 0x0648, 0x0654}, {0x0625, 0x0627, 0x0655}, 
	{0x0626, 0x064A, 0x0654}, {0x06C0, 0x06D5, 0x0654}, 
	{0x06C2, 0x06C1, 0x0654}, {0x06D3, 0x06D2, 0x0654}, 
	{0x0929, 0x0928, 0x093C}, {0x0931, 0x0930, 0x093C}, 
	{0x0934, 0x0933, 0x093C}, {0x0958, 0x0915, 0x093C}, 
	{0x0959, 0x0916, 0x093C}, {0x095A, 0x0917, 0x093C}, 
	{0x095B, 0x091C, 0x093C}, {0x095C, 0x0921, 0x093C}, 
	{0x095D, 0x0922, 0x093C}, {0x095E, 0x092B, 0x093C}, 
	{0x095F, 0x092F, 0x093C}, {0x09DC, 0x09A1, 0x09BC}, 
	{0x09DD, 0x09A2, 0x09BC}, {0x09DF, 0x09AF, 0x09BC}, 
	{0x0A33, 0x0A32, 0x0A3C}, {0x0A36, 0x0A38, 0x0A3C}, 
	{0x0A59, 0x0A16, 0x0A3C}, {0x0A5A, 0x0A17, 0x0A3C}, 
	{0x0A5B, 0x0A1C, 0x0A3C}, {0x0A5E, 0x0A2B, 0x0A3C}, 
	{0x0B5C, 0x0B21, 0x0B3C}, {0x0B5D, 0x0B22, 0x0B3C}, 
	{0x0B94, 0x0B92, 0x0BD7}, {0x0F43, 0x0F42, 0x0FB7}, 
	{0x0F4D, 0x0F4C, 0x0FB7}, {0x0F52, 0x0F51, 0x0FB7}, 
	{0x0F57, 0x0F56, 0x0FB7}, {0x0F5C, 0x0F5B, 0x0FB7}, 
	{0x0F69, 0x0F40, 0x0FB5}, {0x1026, 0x1025, 0x102E}, 
	{0x1E00, 0x0041, 0x0325}, {0x1E01, 0x0061, 0x0325}, 
	{0x1E02, 0x0042, 0x0307}, {0x1E03, 0x0062, 0x0307}, 
	{0x1E04, 0x0042, 0x0323}, {0x1E05, 0x0062, 0x0323}, 
	{0x1E06, 0x0042, 0x0331}, {0x1E07, 0x0062, 0x0331}, 
	{0x1E08, 0x00C7, 0x0301}, {0x1E09, 0x00E7, 0x0301}, 
	{0x1E0A, 0x0044, 0x0307}, {0x1E0B, 0x0064, 0x0307}, 
	{0x1E0C, 0x0044, 0x0323}, {0x1E0D, 0x0064, 0x0323}, 
	{0x1E0E, 0x0044, 0x0331}, {0x1E0F, 0x0064, 0x0331}, 
	{0x1E10, 0x0044, 0x0327}, {0x1E11, 0x0064, 0x0327}, 
	{0x1E12, 0x0044, 0x032D}, {0x1E13, 0x0064, 0x032D}, 
	{0x1E14, 0x0112, 0x0300}, {0x1E15, 0x0113, 0x0300}, 
	{0x1E16, 0x0112, 0x0301}, {0x1E17, 0x0113, 0x0301}, 
	{0x1E18, 0x0045, 0x032D}, {0x1E19, 0x0065, 0x032D}, 
	{0x1E1A, 0x0045, 0x0330}, {0x1E1B, 0x0065, 0x0330}, 
	{0x1E1C, 0x0228, 0x0306}, {0x1E1D, 0x0229, 0x0306}, 
	{0x1E1E, 0x0046, 0x0307}, {0x1E1F, 0x0066, 0x0307}, 
	{0x1E20, 0x0047, 0x0304}, {0x1E21, 0x0067, 0x0304}, 
	{0x1E22, 0x0048, 0x0307}, {0x1E23, 0x0068, 0x0307}, 
	{0x1E24, 0x0048, 0x0323}, {0x1E25, 0x0068, 0x0323}, 
	{0x1E26, 0x0048, 0x0308}, {0x1E27, 0x0068, 0x0308}, 
	{0x1E28, 0x0048, 0x0327}, {0x1E29, 0x0068, 0x0327}, 
	{0x1E2A, 0x0048, 0x032E}, {0x1E2B, 0x0068, 0x032E}, 
	{0x1E2C, 0x0049, 0x0330}, {0x1E2D, 0x0069, 0x0330}, 
	{0x1E2E, 0x00CF, 0x0301}, {0x1E2F, 0x00EF, 0x0301}, 
	{0x1E30, 0x004B, 0x0301}, {0x1E31, 0x006B, 0x0301}, 
	{0x1E32, 0x004B, 0x0323}, {0x1E33, 0x006B, 0x0323}, 
	{0x1E34, 0x004B, 0x0331}, {0x1E35, 0x006B, 0x0331}, 
	{0x1E36, 0x004C, 0x0323}, {0x1E37, 0x006C, 0x0323}, 
	{0x1E38, 0x1E36, 0x0304}, {0x1E39, 0x1E37, 0x0304}, 
	{0x1E3A, 0x004C, 0x0331}, {0x1E3B, 0x006C, 0x0331}, 
	{0x1E3C, 0x004C, 0x032D}, {0x1E3D, 0x006C, 0x032D}, 
	{0x1E3E, 0x004D, 0x0301}, {0x1E3F, 0x006D, 0x0301}, 
	{0x1E40, 0x004D, 0x0307}, {0x1E41, 0x006D, 0x0307}, 
	{0x1E42, 0x004D, 0x0323}, {0x1E43, 0x006D, 0x0323}, 
	{0x1E44, 0x004E, 0x0307}, {0x1E45, 0x006E, 0x0307}, 
	{0x1E46, 0x004E, 0x0323}, {0x1E47, 0x006E, 0x0323}, 
	{0x1E48, 0x004E, 0x0331}, {0x1E49, 0x006E, 0x0331}, 
	{0x1E4A, 0x004E, 0x032D}, {0x1E4B, 0x006E, 0x032D}, 
	{0x1E4C, 0x00D5, 0x0301}, {0x1E4D, 0x00F5, 0x0301}, 
	{0x1E4E, 0x00D5, 0x0308}, {0x1E4F, 0x00F5, 0x0308}, 
	{0x1E50, 0x014C, 0x0300}, {0x1E51, 0x014D, 0x0300}, 
	{0x1E52, 0x014C, 0x0301}, {0x1E53, 0x014D, 0x0301}, 
	{0x1E54, 0x0050, 0x0301}, {0x1E55, 0x0070, 0x0301}, 
	{0x1E56, 0x0050, 0x0307}, {0x1E57, 0x0070, 0x0307}, 
	{0x1E58, 0x0052, 0x0307}, {0x1E59, 0x0072, 0x0307}, 
	{0x1E5A, 0x0052, 0x0323}, {0x1E5B, 0x0072, 0x0323}, 
	{0x1E5C, 0x1E5A, 0x0304}, {0x1E5D, 0x1E5B, 0x0304}, 
	{0x1E5E, 0x0052, 0x0331}, {0x1E5F, 0x0072, 0x0331}, 
	{0x1E60, 0x0053, 0x0307}, {0x1E61, 0x0073, 0x0307}, 
	{0x1E62, 0x0053, 0x0323}, {0x1E63, 0x0073, 0x0323}, 
	{0x1E64, 0x015A, 0x0307}, {0x1E65, 0x015B, 0x0307}, 
	{0x1E66, 0x0160, 0x0307}, {0x1E67, 0x0161, 0x0307}, 
	{0x1E68, 0x1E62, 0x0307}, {0x1E69, 0x1E63, 0x0307}, 
	{0x1E6A, 0x0054, 0x0307}, {0x1E6B, 0x0074, 0x0307}, 
	{0x1E6C, 0x0054, 0x0323}, {0x1E6D, 0x0074, 0x0323}, 
	{0x1E6E, 0x0054, 0x0331}, {0x1E6F, 0x0074, 0x0331}, 
	{0x1E70, 0x0054, 0x032D}, {0x1E71, 0x0074, 0x032D}, 
	{0x1E72, 0x0055, 0x0324}, {0x1E73, 0x0075, 0x0324}, 
	{0x1E74, 0x0055, 0x0330}, {0x1E75, 0x0075, 0x0330}, 
	{0x1E76, 0x0055, 0x032D}, {0x1E77, 0x0075, 0x032D}, 
	{0x1E78, 0x0168, 0x0301}, {0x1E79, 0x0169, 0x0301}, 
	{0x1E7A, 0x016A, 0x0308}, {0x1E7B, 0x016B, 0x0308}, 
	{0x1E7C, 0x0056, 0x0303}, {0x1E7D, 0x0076, 0x0303}, 
	{0x1E7E, 0x0056, 0x0323}, {0x1E7F, 0x0076, 0x0323}, 
	{0x1E80, 0x0057, 0x0300}, {0x1E81, 0x0077, 0x0300}, 
	{0x1E82, 0x0057, 0x0301}, {0x1E83, 0x0077, 0x0301}, 
	{0x1E84, 0x0057, 0x0308}, {0x1E85, 0x0077, 0x0308}, 
	{0x1E86, 0x0057, 0x0307}, {0x1E87, 0x0077, 0x0307}, 
	{0x1E88, 0x0057, 0x0323}, {0x1E89, 0x0077, 0x0323}, 
	{0x1E8A, 0x0058, 0x0307}, {0x1E8B, 0x0078, 0x0307}, 
	{0x1E8C, 0x0058, 0x0308}, {0x1E8D, 0x0078, 0x0308}, 
	{0x1E8E, 0x0059, 0x0307}, {0x1E8F, 0x0079, 0x0307}, 
	{0x1E90, 0x005A, 0x0302}, {0x1E91, 0x007A, 0x0302}, 
	{0x1E92, 0x005A, 0x0323}, {0x1E93, 0x007A, 0x0323}, 
	{0x1E94, 0x005A, 0x0331}, {0x1E95, 0x007A, 0x0331}, 
	{0x1E96, 0x0068, 0x0331}, {0x1E97, 0x0074, 0x0308}, 
	{0x1E98, 0x0077, 0x030A}, {0x1E99, 0x0079, 0x030A}, 
	{0x1E9B, 0x017F, 0x0307}, {0x1EA0, 0x0041, 0x0323}, 
	{0x1EA1, 0x0061, 0x0323}, {0x1EA2, 0x0041, 0x0309}, 
	{0x1EA3, 0x0061, 0x0309}, {0x1EA4, 0x00C2, 0x0301}, 
	{0x1EA5, 0x00E2, 0x0301}, {0x1EA6, 0x00C2, 0x0300}, 
	{0x1EA7, 0x00E2, 0x0300}, {0x1EA8, 0x00C2, 0x0309}, 
	{0x1EA9, 0x00E2, 0x0309}, {0x1EAA, 0x00C2, 0x0303}, 
	{0x1EAB, 0x00E2, 0x0303}, {0x1EAC, 0x1EA0, 0x0302}, 
	{0x1EAD, 0x1EA1, 0x0302}, {0x1EAE, 0x0102, 0x0301}, 
	{0x1EAF, 0x0103, 0x0301}, {0x1EB0, 0x0102, 0x0300}, 
	{0x1EB1, 0x0103, 0x0300}, {0x1EB2, 0x0102, 0x0309}, 
	{0x1EB3, 0x0103, 0x0309}, {0x1EB4, 0x0102, 0x0303}, 
	{0x1EB5, 0x0103, 0x0303}, {0x1EB6, 0x1EA0, 0x0306}, 
	{0x1EB7, 0x1EA1, 0x0306}, {0x1EB8, 0x0045, 0x0323}, 
	{0x1EB9, 0x0065, 0x0323}, {0x1EBA, 0x0045, 0x0309}, 
	{0x1EBB, 0x0065, 0x0309}, {0x1EBC, 0x0045, 0x0303}, 
	{0x1EBD, 0x0065, 0x0303}, {0x1EBE, 0x00CA, 0x0301}, 
	{0x1EBF, 0x00EA, 0x0301}, {0x1EC0, 0x00CA, 0x0300}, 
	{0x1EC1, 0x00EA, 0x0300}, {0x1EC2, 0x00CA, 0x0309}, 
	{0x1EC3, 0x00EA, 0x0309}, {0x1EC4, 0x00CA, 0x0303}, 
	{0x1EC5, 0x00EA, 0x0303}, {0x1EC6, 0x1EB8, 0x0302}, 
	{0x1EC7, 0x1EB9, 0x0302}, {0x1EC8, 0x0049, 0x0309}, 
	{0x1EC9, 0x0069, 0x0309}, {0x1ECA, 0x0049, 0x0323}, 
	{0x1ECB, 0x0069, 0x0323}, {0x1ECC, 0x004F, 0x0323}, 
	{0x1ECD, 0x006F, 0x0323}, {0x1ECE, 0x004F, 0x0309}, 
	{0x1ECF, 0x006F, 0x0309}, {0x1ED0, 0x00D4, 0x0301}, 
	{0x1ED1, 0x00F4, 0x0301}, {0x1ED2, 0x00D4, 0x0300}, 
	{0x1ED3, 0x00F4, 0x0300}, {0x1ED4, 0x00D4, 0x0309}, 
	{0x1ED5, 0x00F4, 0x0309}, {0x1ED6, 0x00D4, 0x0303}, 
	{0x1ED7, 0x00F4, 0x0303}, {0x1ED8, 0x1ECC, 0x0302}, 
	{0x1ED9, 0x1ECD, 0x0302}, {0x1EDA, 0x01A0, 0x0301}, 
	{0x1EDB, 0x01A1, 0x0301}, {0x1EDC, 0x01A0, 0x0300}, 
	{0x1EDD, 0x01A1, 0x0300}, {0x1EDE, 0x01A0, 0x0309}, 
	{0x1EDF, 0x01A1, 0x0309}, {0x1EE0, 0x01A0, 0x0303}, 
	{0x1EE1, 0x01A1, 0x0303}, {0x1EE2, 0x01A0, 0x0323}, 
	{0x1EE3, 0x01A1, 0x0323}, {0x1EE4, 0x0055, 0x0323}, 
	{0x1EE5, 0x0075, 0x0323}, {0x1EE6, 0x0055, 0x0309}, 
	{0x1EE7, 0x0075, 0x0309}, {0x1EE8, 0x01AF, 0x0301}, 
	{0x1EE9, 0x01B0, 0x0301}, {0x1EEA, 0x01AF, 0x0300}, 
	{0x1EEB, 0x01B0, 0x0300}, {0x1EEC, 0x01AF, 0x0309}, 
	{0x1EED, 0x01B0, 0x0309}, {0x1EEE, 0x01AF, 0x0303}, 
	{0x1EEF, 0x01B0, 0x0303}, {0x1EF0, 0x01AF, 0x0323}, 
	{0x1EF1, 0x01B0, 0x0323}, {0x1EF2, 0x0059, 0x0300}, 
	{0x1EF3, 0x0079, 0x0300}, {0x1EF4, 0x0059, 0x0323}, 
	{0x1EF5, 0x0079, 0x0323}, {0x1EF6, 0x0059, 0x0309}, 
	{0x1EF7, 0x0079, 0x0309}, {0x1EF8, 0x0059, 0x0303}, 
	{0x1EF9, 0x0079, 0x0303}, {0x1F00, 0x03B1, 0x0313}, 
	{0x1F01, 0x03B1, 0x0314}, {0x1F02, 0x1F00, 0x0300}, 
	{0x1F03, 0x1F01, 0x0300}, {0x1F04, 0x1F00, 0x0301}, 
	{0x1F05, 0x1F01, 0x0301}, {0x1F06, 0x1F00, 0x0342}, 
	{0x1F07, 0x1F01, 0x0342}, {0x1F08, 0x0391, 0x0313}, 
	{0x1F09, 0x0391, 0x0314}, {0x1F0A, 0x1F08, 0x0300}, 
	{0x1F0B, 0x1F09, 0x0300}, {0x1F0C, 0x1F08, 0x0301}, 
	{0x1F0D, 0x1F09, 0x0301}, {0x1F0E, 0x1F08, 0x0342}, 
	{0x1F0F, 0x1F09, 0x0342}, {0x1F10, 0x03B5, 0x0313}, 
	{0x1F11, 0x03B5, 0x0314}, {0x1F12, 0x1F10, 0x0300}, 
	{0x1F13, 0x1F11, 0x0300}, {0x1F14, 0x1F10, 0x0301}, 
	{0x1F15, 0x1F11, 0x0301}, {0x1F18, 0x0395, 0x0313}, 
	{0x1F19, 0x0395, 0x0314}, {0x1F1A, 0x1F18, 0x0300}, 
	{0x1F1B, 0x1F19, 0x0300}, {0x1F1C, 0x1F18, 0x0301}, 
	{0x1F1D, 0x1F19, 0x0301}, {0x1F20, 0x03B7, 0x0313}, 
	{0x1F21, 0x03B7, 0x0314}, {0x1F22, 0x1F20, 0x0300}, 
	{0x1F23, 0x1F21, 0x0300}, {0x1F24, 0x1F20, 0x0301}, 
	{0x1F25, 0x1F21, 0x0301}, {0x1F26, 0x1F20, 0x0342}, 
	{0x1F27, 0x1F21, 0x0342}, {0x1F28, 0x0397, 0x0313}, 
	{0x1F29, 0x0397, 0x0314}, {0x1F2A, 0x1F28, 0x0300}, 
	{0x1F2B, 0x1F29, 0x0300}, {0x1F2C, 0x1F28, 0x0301}, 
	{0x1F2D, 0x1F29, 0x0301}, {0x1F2E, 0x1F28, 0x0342}, 
	{0x1F2F, 0x1F29, 0x0342}, {0x1F30, 0x03B9, 0x0313}, 
	{0x1F31, 0x03B9, 0x0314}, {0x1F32, 0x1F30, 0x0300}, 
	{0x1F33, 0x1F31, 0x0300}, {0x1F34, 0x1F30, 0x0301}, 
	{0x1F35, 0x1F31, 0x0301}, {0x1F36, 0x1F30, 0x0342}, 
	{0x1F37, 0x1F31, 0x0342}, {0x1F38, 0x0399, 0x0313}, 
	{0x1F39, 0x0399, 0x0314}, {0x1F3A, 0x1F38, 0x0300}, 
	{0x1F3B, 0x1F39, 0x0300}, {0x1F3C, 0x1F38, 0x0301}, 
	{0x1F3D, 0x1F39, 0x0301}, {0x1F3E, 0x1F38, 0x0342}, 
	{0x1F3F, 0x1F39, 0x0342}, {0x1F40, 0x03BF, 0x0313}, 
	{0x1F41, 0x03BF, 0x0314}, {0x1F42, 0x1F40, 0x0300}, 
	{0x1F43, 0x1F41, 0x0300}, {0x1F44, 0x1F40, 0x0301}, 
	{0x1F45, 0x1F41, 0x0301}, {0x1F48, 0x039F, 0x0313}, 
	{0x1F49, 0x039F, 0x0314}, {0x1F4A, 0x1F48, 0x0300}, 
	{0x1F4B, 0x1F49, 0x0300}, {0x1F4C, 0x1F48, 0x0301}, 
	{0x1F4D, 0x1F49, 0x0301}, {0x1F50, 0x03C5, 0x0313}, 
	{0x1F51, 0x03C5, 0x0314}, {0x1F52, 0x1F50, 0x0300}, 
	{0x1F53, 0x1F51, 0x0300}, {0x1F54, 0x1F50, 0x0301}, 
	{0x1F55, 0x1F51, 0x0301}, {0x1F56, 0x1F50, 0x0342}, 
	{0x1F57, 0x1F51, 0x0342}, {0x1F59, 0x03A5, 0x0314}, 
	{0x1F5B, 0x1F59, 0x0300}, {0x1F5D, 0x1F59, 0x0301}, 
	{0x1F5F, 0x1F59, 0x0342}, {0x1F60, 0x03C9, 0x0313}, 
	{0x1F61, 0x03C9, 0x0314}, {0x1F62, 0x1F60, 0x0300}, 
	{0x1F63, 0x1F61, 0x0300}, {0x1F64, 0x1F60, 0x0301}, 
	{0x1F65, 0x1F61, 0x0301}, {0x1F66, 0x1F60, 0x0342}, 
	{0x1F67, 0x1F61, 0x0342}, {0x1F68, 0x03A9, 0x0313}, 
	{0x1F69, 0x03A9, 0x0314}, {0x1F6A, 0x1F68, 0x0300}, 
	{0x1F6B, 0x1F69, 0x0300}, {0x1F6C, 0x1F68, 0x0301}, 
	{0x1F6D, 0x1F69, 0x0301}, {0x1F6E, 0x1F68, 0x0342}, 
	{0x1F6F, 0x1F69, 0x0342}, {0x1F70, 0x03B1, 0x0300}, 
	{0x1F72, 0x03B5, 0x0300}, {0x1F74, 0x03B7, 0x0300}, 
	{0x1F76, 0x03B9, 0x0300}, {0x1F78, 0x03BF, 0x0300}, 
	{0x1F7A, 0x03C5, 0x0300}, {0x1F7C, 0x03C9, 0x0300}, 
	{0x1F80, 0x1F00, 0x0345}, {0x1F81, 0x1F01, 0x0345}, 
	{0x1F82, 0x1F02, 0x0345}, {0x1F83, 0x1F03, 0x0345}, 
	{0x1F84, 0x1F04, 0x0345}, {0x1F85, 0x1F05, 0x0345}, 
	{0x1F86, 0x1F06, 0x0345}, {0x1F87, 0x1F07, 0x0345}, 
	{0x1F88, 0x1F08, 0x0345}, {0x1F89, 0x1F09, 0x0345}, 
	{0x1F8A, 0x1F0A, 0x0345}, {0x1F8B, 0x1F0B, 0x0345}, 
	{0x1F8C, 0x1F0C, 0x0345}, {0x1F8D, 0x1F0D, 0x0345}, 
	{0x1F8E, 0x1F0E, 0x0345}, {0x1F8F, 0x1F0F, 0x0345}, 
	{0x1F90, 0x1F20, 0x0345}, {0x1F91, 0x1F21, 0x0345}, 
	{0x1F92, 0x1F22, 0x0345}, {0x1F93, 0x1F23, 0x0345}, 
	{0x1F94, 0x1F24, 0x0345}, {0x1F95, 0x1F25, 0x0345}, 
	{0x1F96, 0x1F26, 0x0345}, {0x1F97, 0x1F27, 0x0345}, 
	{0x1F98, 0x1F28, 0x0345}, {0x1F99, 0x1F29, 0x0345}, 
	{0x1F9A, 0x1F2A, 0x0345}, {0x1F9B, 0x1F2B, 0x0345}, 
	{0x1F9C, 0x1F2C, 0x0345}, {0x1F9D, 0x1F2D, 0x0345}, 
	{0x1F9E, 0x1F2E, 0x0345}, {0x1F9F, 0x1F2F, 0x0345}, 
	{0x1FA0, 0x1F60, 0x0345}, {0x1FA1, 0x1F61, 0x0345}, 
	{0x1FA2, 0x1F62, 0x0345}, {0x1FA3, 0x1F63, 0x0345}, 
	{0x1FA4, 0x1F64, 0x0345}, {0x1FA5, 0x1F65, 0x0345}, 
	{0x1FA6, 0x1F66, 0x0345}, {0x1FA7, 0x1F67, 0x0345}, 
	{0x1FA8, 0x1F68, 0x0345}, {0x1FA9, 0x1F69, 0x0345}, 
	{0x1FAA, 0x1F6A, 0x0345}, {0x1FAB, 0x1F6B, 0x0345}, 
	{0x1FAC, 0x1F6C, 0x0345}, {0x1FAD, 0x1F6D, 0x0345}, 
	{0x1FAE, 0x1F6E, 0x0345}, {0x1FAF, 0x1F6F, 0x0345}, 
	{0x1FB0, 0x03B1, 0x0306}, {0x1FB1, 0x03B1, 0x0304}, 
	{0x1FB2, 0x1F70, 0x0345}, {0x1FB3, 0x03B1, 0x0345}, 
	{0x1FB4, 0x03AC, 0x0345}, {0x1FB6, 0x03B1, 0x0342}, 
	{0x1FB7, 0x1FB6, 0x0345}, {0x1FB8, 0x0391, 0x0306}, 
	{0x1FB9, 0x0391, 0x0304}, {0x1FBA, 0x0391, 0x0300}, 
	{0x1FBC, 0x0391, 0x0345}, {0x1FC2, 0x1F74, 0x0345}, 
	{0x1FC3, 0x03B7, 0x0345}, {0x1FC4, 0x03AE, 0x0345}, 
	{0x1FC6, 0x03B7, 0x0342}, {0x1FC7, 0x1FC6, 0x0345}, 
	{0x1FC8, 0x0395, 0x0300}, {0x1FCA, 0x0397, 0x0300}, 
	{0x1FCC, 0x0397, 0x0345}, {0x1FD0, 0x03B9, 0x0306}, 
	{0x1FD1, 0x03B9, 0x0304}, {0x1FD2, 0x03CA, 0x0300}, 
	{0x1FD6, 0x03B9, 0x0342}, {0x1FD7, 0x03CA, 0x0342}, 
	{0x1FD8, 0x0399, 0x0306}, {0x1FD9, 0x0399, 0x0304}, 
	{0x1FDA, 0x0399, 0x0300}, {0x1FE0, 0x03C5, 0x0306}, 
	{0x1FE1, 0x03C5, 0x0304}, {0x1FE2, 0x03CB, 0x0300}, 
	{0x1FE4, 0x03C1, 0x0313}, {0x1FE5, 0x03C1, 0x0314}, 
	{0x1FE6, 0x03C5, 0x0342}, {0x1FE7, 0x03CB, 0x0342}, 
	{0x1FE8, 0x03A5, 0x0306}, {0x1FE9, 0x03A5, 0x0304}, 
	{0x1FEA, 0x03A5, 0x0300}, {0x1FEC, 0x03A1, 0x0314}, 
	{0x1FF2, 0x1F7C, 0x0345}, {0x1FF3, 0x03C9, 0x0345}, 
	{0x1FF4, 0x03CE, 0x0345}, {0x1FF6, 0x03C9, 0x0342}, 
	{0x1FF7, 0x1FF6, 0x0345}, {0x1FF8, 0x039F, 0x0300}, 
	{0x1FFA, 0x03A9, 0x0300}, {0x1FFC, 0x03A9, 0x0345}, 
	{0x304C, 0x304B, 0x3099}, {0x304E, 0x304D, 0x3099}, 
	{0x3050, 0x304F, 0x3099}, {0x3052, 0x3051, 0x3099}, 
	{0x3054, 0x3053, 0x3099}, {0x3056, 0x3055, 0x3099}, 
	{0x3058, 0x3057, 0x3099}, {0x305A, 0x3059, 0x3099}, 
	{0x305C, 0x305B, 0x3099}, {0x305E, 0x305D, 0x3099}, 
	{0x3060, 0x305F, 0x3099}, {0x3062, 0x3061, 0x3099}, 
	{0x3065, 0x3064, 0x3099}, {0x3067, 0x3066, 0x3099}, 
	{0x3069, 0x3068, 0x3099}, {0x3070, 0x306F, 0x3099}, 
	{0x3071, 0x306F, 0x309A}, {0x3073, 0x3072, 0x3099}, 
	{0x3074, 0x3072, 0x309A}, {0x3076, 0x3075, 0x3099}, 
	{0x3077, 0x3075, 0x309A}, {0x3079, 0x3078, 0x3099}, 
	{0x307A, 0x3078, 0x309A}, {0x307C, 0x307B, 0x3099}, 
	{0x307D, 0x307B, 0x309A}, {0x3094, 0x3046, 0x3099}, 
	{0x309E, 0x309D, 0x3099}, {0x30AC, 0x30AB, 0x3099}, 
	{0x30AE, 0x30AD, 0x3099}, {0x30B0, 0x30AF, 0x3099}, 
	{0x30B2, 0x30B1, 0x3099}, {0x30B4, 0x30B3, 0x3099}, 
	{0x30B6, 0x30B5, 0x3099}, {0x30B8, 0x30B7, 0x3099}, 
	{0x30BA, 0x30B9, 0x3099}, {0x30BC, 0x30BB, 0x3099}, 
	{0x30BE, 0x30BD, 0x3099}, {0x30C0, 0x30BF, 0x3099}, 
	{0x30C2, 0x30C1, 0x3099}, {0x30C5, 0x30C4, 0x3099}, 
	{0x30C7, 0x30C6, 0x3099}, {0x30C9, 0x30C8, 0x3099}, 
	{0x30D0, 0x30CF, 0x3099}, {0x30D1, 0x30CF, 0x309A}, 
	{0x30D3, 0x30D2, 0x3099}, {0x30D4, 0x30D2, 0x309A}, 
	{0x30D6, 0x30D5, 0x3099}, {0x30D7, 0x30D5, 0x309A}, 
	{0x30D9, 0x30D8, 0x3099}, {0x30DA, 0x30D8, 0x309A}, 
	{0x30DC, 0x30DB, 0x3099}, {0x30DD, 0x30DB, 0x309A}, 
	{0x30F4, 0x30A6, 0x3099}, {0x30F7, 0x30EF, 0x3099}, 
	{0x30F8, 0x30F0, 0x3099}, {0x30F9, 0x30F1, 0x3099}, 
	{0x30FA, 0x30F2, 0x3099}, {0x30FE, 0x30FD, 0x3099}, 
	{0xFB1D, 0x05D9, 0x05B4}, {0xFB1F, 0x05F2, 0x05B7}, 
	{0xFB2A, 0x05E9, 0x05C1}, {0xFB2B, 0x05E9, 0x05C2}, 
	{0xFB2C, 0xFB49, 0x05C1}, {0xFB2D, 0xFB49, 0x05C2}, 
	{0xFB2E, 0x05D0, 0x05B7}, {0xFB2F, 0x05D0, 0x05B8}, 
	{0xFB30, 0x05D0, 0x05BC}, {0xFB31, 0x05D1, 0x05BC}, 
	{0xFB32, 0x05D2, 0x05BC}, {0xFB33, 0x05D3, 0x05BC}, 
	{0xFB34, 0x05D4, 0x05BC}, {0xFB35, 0x05D5, 0x05BC}, 
	{0xFB36, 0x05D6, 0x05BC}, {0xFB38, 0x05D8, 0x05BC}, 
	{0xFB39, 0x05D9, 0x05BC}, {0xFB3A, 0x05DA, 0x05BC}, 
	{0xFB3B, 0x05DB, 0x05BC}, {0xFB3C, 0x05DC, 0x05BC}, 
	{0xFB3E, 0x05DE, 0x05BC}, {0xFB40, 0x05E0, 0x05BC}, 
	{0xFB41, 0x05E1, 0x05BC}, {0xFB43, 0x05E3, 0x05BC}, 
	{0xFB44, 0x05E4, 0x05BC}, {0xFB46, 0x05E6, 0x05BC}, 
	{0xFB47, 0x05E7, 0x05BC}, {0xFB48, 0x05E8, 0x05BC}, 
	{0xFB49, 0x05E9, 0x05BC}, {0xFB4A, 0x05EA, 0x05BC}, 
	{0xFB4B, 0x05D5, 0x05B9}, {0xFB4C, 0x05D1, 0x05BF}, 
	{0xFB4D, 0x05DB, 0x05BF}, {0xFB4E, 0x05E4, 0x05BF}, 
}; // precomposeTable

// Return index in table if "c" is in "table[size]", else -1.  
static int inPrecomposeTable(wide_t c) {
    int mid, bot, top;
    /* first quick check for Latin1 etc. characters */
    if (c < precomposeTable[0].ucsChar) return -1;
    /* binary search in table */
    bot = 0;
    top = (sizeof(precomposeTable)/sizeof(precomposeStruct)) - 1;
    while (top >= bot) {
		mid = (bot + top) >> 1;
		if (precomposeTable[mid].ucsChar < c)
			bot = mid + 1;
		else if (precomposeTable[mid].ucsChar > c)
			top = mid - 1;
		else
			return mid;
	}
	return -1;
} // inPrecomposeTable

void unPrecompose(wide_t *dest, int *destLength, const wide_t *source,
		int sourceLength) {
	const wide_t *sourcePtr;
	wide_t *outPtr;
	int charCount;
	for (charCount = 0, sourcePtr = source, outPtr = dest;
			charCount < sourceLength; charCount += 1) {
		// copy one wide character if it is not combining
		int index = inPrecomposeTable(*sourcePtr);
		if (index == -1) { // not precomposed
			*outPtr++ = *sourcePtr;
		} else {
			*outPtr++ = precomposeTable[index].first;
			*outPtr++ = precomposeTable[index].second;
		}
		sourcePtr++;
	}
	*destLength = outPtr - dest;
} // unPrecompose

// Table of conversions from non-final to final forms of letters.  Generated by
// a script from the Unicode 4.0 table.
tableEntry toFinalTable[] = {
	{0x03C3, 0x03C3, 0x03C2},
	{0x05DB, 0x05DB, 0x05DA},
	{0x05DE, 0x05DE, 0x05DD},
	{0x05E0, 0x05E0, 0x05DF},
	{0x05E4, 0x05E4, 0x05E3},
	{0x05E6, 0x05E6, 0x05E5},
	{0x0723, 0x0723, 0x0724},
	{0x3106, 0x3106, 0x31B4},
	{0x310A, 0x310A, 0x31B5},
	{0x310E, 0x310E, 0x31B6},
	{0x310F, 0x310F, 0x31B7},
	{0xFB3B, 0xFB3B, 0xFB3A},
	{0xFB44, 0xFB44, 0xFB43},
};

wide_t toFinal(wide_t c) {
	int index = inTable(toFinalTable,
		sizeof(toFinalTable) / sizeof(tableEntry), c);
	if (index == -1)
		return(c);
	else
		return(toFinalTable[index].replacement);
} // toFinal
