/*
 * gridops.c
 * copyright (c) 2004 Wei-Keat Kong.
 *
 * Functions for manipulating the dictionary.
 *
 * FindSlotListIdxFromCell: Locates slotlist index based on current cell
 * FindConstraintForSlotList: Searches slotlist for constraints
 * InsertWordGrid: Insert a word into the grid
 * RemoveWordGrid: Remove a word from the grid
 * LogAnswer: Save the grid to file
 * PrintAnswer: Print the current state of the grid
 * get_gridCMD: Tk interface to retrieve the contents of a cell.
*/

#include "gridops.h"

/*
 * Function: FindSlotListIdxFromCell
 * Purpose: Find the index of the slotlist related to the cell. Depends on
 *          direction
 * I/O: In  - slotlist, x, y, index, position, direction
 *      Out - index, position
 * Returns: RC_OK/RC_FAILED
 * Functions called: None.
 * Algorithm: For each word in the slotlist, check to see if the current
 *            cell is within the length of the word in the slotlist.
*/
int FindSlotListIdxFromCell(slot_list_t *slotlist,
                             u_short x, u_short y, u_short *index,
                             u_short *wordpos, u_short direction)
{
   int i=0; // counter for the number of words in the slotlist

   for(i=0;i<NUM_WORDS;i++)
   {
      if (direction == ACROSS)
      {
         // Check if we're in the right row, the column must be within
         // the length of the word, and the direction is correct

         if((slotlist[i].h_idx == x)&&
            (y <= slotlist[i].w_idx + slotlist[i].length)&&
            (slotlist[i].direction == ACROSS))
         {
            *index = i;
            *wordpos = y - slotlist[i].w_idx;
            return RC_OK;
         }
      }
      else
      {
         // Check if we're in the right column, the row must be within
         // the length of the word, ant the direction is correct

         if((slotlist[i].w_idx == y)&&
            (x <= slotlist[i].h_idx + slotlist[i].length)&&
            (slotlist[i].direction == DOWN))
         {
            *index = i;
            *wordpos = x - slotlist[i].h_idx;
            return RC_OK;
         }
      }
   }

   return RC_FAILED;
}

/*
 * Function: FindConstraintForSlotList
 * Purpose: Fill out the constraint structure for each word in the slotlist.
 * I/O: In  - slotlist
 *      Out - slotlist
 * Returns: none.
 * Functions called: FindSlotListIdxFromCell()
 * Algorithm: For each word in the slotlist, identify constraints with other
 *            words in the slotlist
*/
void FindConstraintForSlotList(slot_list_t *slotlist)
{
   int i=0; // Counter for number of words in the slotlist
   int s_x=0, s_y=0, c_idx=0; // Counter for current x, y, and index
   u_short index=0, wordpos=0;

   for(i=0;i<NUM_WORDS;i++)
   {
      //fprintf(stderr, "%d,%d dir=%d len=%d ", slotlist[i].h_idx,
      //        slotlist[i].w_idx, slotlist[i].direction,
      //        slotlist[i].length);
      c_idx = 0;
      if (slotlist[i].direction == ACROSS)
      {
         for(s_y=slotlist[i].w_idx;
             s_y<slotlist[i].w_idx+slotlist[i].length;s_y++)
         {
            // For each cell in the current word, find the index of the
            // word that intersects.

            if(grid[slotlist[i].h_idx][s_y].constrained)
            {
               FindSlotListIdxFromCell(slotlist,
                                       slotlist[i].h_idx, s_y,
                                       &index, &wordpos, DOWN);
               //fprintf(stderr, "h_idx=%d, s_y=%d, ACROSS, idx=%d, post=%d\n",
               //          slotlist[i].h_idx, s_y, index, wordpos);
               slotlist[i].constraint[c_idx].idx1 = i;
               slotlist[i].constraint[c_idx].pos1 = s_y - slotlist[i].w_idx;
               slotlist[i].constraint[c_idx].idx2 = index;
               slotlist[i].constraint[c_idx].pos2 = wordpos;
               slotlist[i].numconstraint++;
               c_idx++;
            }
         }
      }
      else
      {
         // Check to make sure direction is valid

         if (slotlist[i].direction != DOWN)
         {
            fprintf(stderr, "Invalid Direction! i=%d", i);
            exit(RC_FAILED);
         }

         for(s_x=slotlist[i].h_idx;
             s_x<slotlist[i].h_idx+slotlist[i].length;s_x++)
         {
            // For each cell in the current word, find the index of the
            // word that intersects.

            if(grid[s_x][slotlist[i].w_idx].constrained)
            {
               FindSlotListIdxFromCell(slotlist,
                                       s_x, slotlist[i].w_idx,
                                       &index, &wordpos, ACROSS);
               //fprintf(stderr, "s_x=%d, w_idx=%d, DOWN, idx=%d, post=%d\n",
               //        s_x, slotlist[i].w_idx, index, wordpos);
               slotlist[i].constraint[c_idx].idx1 = i;
               slotlist[i].constraint[c_idx].pos1 = s_x - slotlist[i].h_idx;
               slotlist[i].constraint[c_idx].idx2 = index;
               slotlist[i].constraint[c_idx].pos2 = wordpos;
               slotlist[i].numconstraint++;
               c_idx++;
            }
         } // End for
      } // End if
   } // End for
}

/*
 * Function: InsertWordGrid()
 * Purpose: inserts the word into the grid at x,y of given length/direction.
 * I/O: In  - string, ucslength, x, y, direction, length
 *      Out - none
 * Returns: none.
 * Functions called: none.
 * Algorithm: Convert the string into characters and then insert them
 *            into the grid, then increasing the exists variable by 1.
 *            This is to keep track of how many words are using this cell.
*/
void InsertWordGrid(wide_t *string, u_short ucslength, u_short x,
                    u_short y, u_short direction, u_short length)
{        
   wide_t wide_word[length][UCSLETTER];
   u_short grid_index=0;
   int ucscnt=0;
   
   if(StringToWideT(wide_word, string, ucslength, length)==RC_FAILED)
   {
      WriteLog("InsertWordGrid: Failed to convert string to wide_t.\n");
      exit(RC_FAILED); 
   }
    
   if(direction==ACROSS)
   {
      //fprintf(stderr, "writing Across %d,%d %d\n", x, y, length);
      for(grid_index=y;grid_index<y+length;grid_index++)
      {
         // Write
         for(ucscnt=0;ucscnt<UCSLETTER;ucscnt++)
         {
			cell_t *gridPtr = &grid[x][grid_index];
            if((gridPtr->letter[ucscnt] == 0)||
               (gridPtr->letter[ucscnt] ==
                wide_word[grid_index-y][ucscnt])) 
            {
               gridPtr->letter[ucscnt] = 
                  wide_word[grid_index-y][ucscnt];
            }
            else
            {
			   fprintf(stderr, "writing Across at (%d,%d) length %d\n",
			   	x, y, length);
               fprintf(stderr, "Tried to insert an invalid word \n");
               exit(RC_FAILED);
            }
         }
         grid[x][grid_index].exists++;
      } // each grid_index across
   } // across index
   else
   { // Down
      // fprintf(stderr, "writing down %d,%d %d\n", x, y, length);
      // Write
      for(grid_index=x;grid_index<x+length;grid_index++)
      {
	  	 cell_t *gridPtr = &grid[grid_index][y];
         for(ucscnt=0;ucscnt<UCSLETTER;ucscnt++)
         {
            if((gridPtr->letter[ucscnt] == 0)||
               (gridPtr->letter[ucscnt] ==
                wide_word[grid_index-x][ucscnt]))
            {
               gridPtr->letter[ucscnt] = 
                  wide_word[grid_index-x][ucscnt];
            }
            else
            {
               fprintf(stderr, "word invalid; ucscnt %d, grid_index %d\n",
			   	ucscnt, grid_index);
			   fprintf(stderr,
			   	"trying to write down starting at (%d, %d) length %d\n",
				 x, y, length);
			   PrintAnswer();
               exit(RC_FAILED);
            }

         }
         grid[grid_index][y].exists++;
      } // each grid index;
   } // Down
} // InsertWordGrid

/*
 * Function: RemoveWordGrid()
 * Purpose: removes the word located at x,y of given length/direction.
 * I/O: In  - x,y, direction, length
 *      Out - none
 * Returns: none.
 * Functions called: none.
 * Algorithm: Check to see if the cell with the character being deleted
 *            is being used by another word. If so, don't delete the 
 *            character but decrease the exists variable by 1.
*/
void RemoveWordGrid(u_short x, u_short y, u_short direction,
                        u_short length)
{   
   u_short grid_index=0;

   // Validate input
   if((direction!=ACROSS)&&(direction!=DOWN))
   {
      fprintf(stderr, "RemoveWord: Invalid direction. %d\n", direction);
      exit(RC_FAILED);
   }
   if((x>=GRIDSIZE)||(y>=GRIDSIZE))
   {
      fprintf(stderr, "RemoveWord: Invalid x,y. %d,%d\n", x,y);
      exit(RC_FAILED);
   } 
   if(length>GRIDSIZE)
   {
      fprintf(stderr, "RemoveWord: Invalid length %d.\n", length);
      exit(RC_FAILED);
   }

   if(direction==ACROSS)
   {
      //fprintf(stderr, "deleting Across %d,%d %d\n", x, y, length);
      for(grid_index=y;grid_index<y+length;grid_index++)
      {
	  	 cell_t *gridPtr = &grid[x][grid_index];
         if(gridPtr->exists>1)
         {
            gridPtr->exists--;
         }
         else
         {
            gridPtr->letter[0] = 0;
            gridPtr->letter[1] = 0;
            gridPtr->letter[2] = 0;
            gridPtr->letter[3] = 0;
            gridPtr->exists = 0;
         }
      } // each grid_index
   } // ACROSS
   else
   {
      //fprintf(stderr, "deleting Down %d,%d %d\n", x, y, length);
      for(grid_index=x;grid_index<x+length;grid_index++)
      {
	  	 cell_t *gridPtr = &grid[grid_index][y];
         if(gridPtr->exists>1) 
         {
            gridPtr->exists--;
         }
         else
         {
            gridPtr->letter[0] = 0;
            gridPtr->letter[1] = 0;
            gridPtr->letter[2] = 0;
            gridPtr->letter[3] = 0;
            gridPtr->exists = 0;
         }
      } // each grid_index
   } // ACROSS
   //PrintAnswer();
} // RemoveWordGrid

/*
 * Function: get_gridCMD
 * Purpose: gets the word from the grid and puts it into the shared 
 *          variable gridstring.
 * I/O: In  - Standard Tk arguments
 *      Out - Standard Tk return codes
 * Returns: TCL_OK
 * Functions called: none
*/
int get_gridCMD(ClientData clientdata, Tcl_Interp *interp, int argc,
                 char *argv[])
{
   int i,j;  // Grid coordinates
   utf8_t *outputstring;
   div_t result;
   int cell=0;
   int num_char;

   outputstring = calloc(MAXSTRINGSIZE, sizeof(utf8_t));
   memset(gridstring, 0 , sizeof(utf8_t)*MAXGRIDSTRING);

   cell = atoi(argv[1]);
   result = div (cell, GRIDSIZE);

   i = result.quot;
   j = result.rem;

   if(grid[i][j].exists!=0)
   {
      num_char = wide_utf8(outputstring, MAXSTRINGSIZE,
                           grid[i][j].letter, UCSLETTER);
      memcpy(gridstring, outputstring, sizeof(utf8_t)*num_char);
   }

   free(outputstring);
   return TCL_OK;
}

/*
 * Function: LogAnswer()
 * Purpose: saves the current state of the grid to the log file
 * I/O: In  - none
 *      Out - none
 * Returns: none.
 * Functions called: none.
*/
void LogAnswer()
{
   int i,j;  // Grid coordinates
   u_short comb_idx=0;
   utf8_t *outputstring;
   int num_char=0;

   outputstring = calloc(MAXSTRINGSIZE, sizeof(utf8_t));

   for(i=0;i<GRIDSIZE;i++) {
      for(j=0;j<GRIDSIZE;j++) {
         if(grid[i][j].isblack) WriteLog("# ");
         else
         {
            if(grid[i][j].exists!=0)
            {
     
               num_char = wide_utf8(outputstring, MAXSTRINGSIZE,
                                    grid[i][j].letter, UCSLETTER);
               //printf("num_char=%d ", num_char);

               /*
               for(comb_idx=0;comb_idx<num_char-1;comb_idx++)
               {
                 WriteLog("%x", outputstring[comb_idx]);
               }
               printf("\n");
               */

               //WriteLog("\\u%.4x", grid[i][j].letter[0]);
               WriteLog("%c", grid[i][j].letter[0]);
               for(comb_idx=1;comb_idx<UCSLETTER;comb_idx++)
               {
                  if(grid[i][j].letter[comb_idx]!=0)
                     WriteLog("%c",(u_int)grid[i][j].letter[comb_idx]);
                   //  WriteLog("\\u%.4x",(u_int)grid[i][j].letter[comb_idx]);
               }

               WriteLog(" ");
            }
            else WriteLog("- ");
         }
      }
      WriteLog("\n");
   } 
   WriteLog("\n");

   free(outputstring);
}

/*
 * Function: PrintAnswer()
 * Purpose: prints the current state of the grid
 * I/O: In  - none
 *      Out - none
 * Returns: none.
 * Functions called: none.
*/
void PrintAnswer()
{    
   int i,j;  // Grid coordinates

   for(i=0;i<GRIDSIZE;i++) {
      for(j=0;j<GRIDSIZE;j++) {
         if(grid[i][j].isblack) fprintf(stdout, "#");
         else
         {
            //  if(grid[i][j].constrained!=0) fprintf(stderr, "*");
            //  else printf("-");
            if(grid[i][j].exists!=0)
            {
               //fprintf(stderr, "%c", (char) grid[i][j].letter[0]);
               fprintf(stdout, "%c", (char) grid[i][j].letter[0]);
            }
            else fprintf(stdout, "-");
            //else fprintf(stderr, "-");
         }
      }
      fprintf(stdout,"\n");
      //fprintf(stderr,"\n");
   }
}
