/*
 * backjump.c
 * copyright (c) 2004 Wei-Keat Kong.
 *
 * Backtrack-related functions.
 *
 * FindNextWord: Function for finding the next word to solve
 * backjumpCMD: Tk interface to backjumping function
 *
*/

#include <sys/time.h>
#include <sys/resource.h>
#include "backjump.h"

/* Function: calculates the difference in time between two time structs.
 * Input: [in]start time, [in]stop time, [out] Result in MS
 * Output: 1.
*/

void timeval_diff(struct timeval start, struct timeval stop, double *TotalTime)
{
   double sec, u_sec;

   sec = stop.tv_sec - start.tv_sec;
   u_sec = stop.tv_usec - start.tv_usec;

   *TotalTime = sec + u_sec/1000000;
}

/*
 * Function:FindNextWord 
 * Purpose: Find the next word in the word list to solve.
 * I/O: In  - slotlist, wordtable, index, dictionary
 *      Out - none
 * Returns: index of the slotlist to be solved
 * Functions called: FindWords()
*/
#define LIMIT 9999
int FindNextWord(slot_list_t *slotlist, int *wtbl, int index, dict_idx *dict)
{
	int i=0, result=LIMIT, resultidx=0;
	int x, y, length, direction;
	// return index; // turn off dynamic reordering.
	// For each word left in the slotlist, find the word with the least number
	// of solutions remaining.
	for(i=index;i<NUM_WORDS;i++)
	{
		slot_list_t *theSlot = &slotlist[wtbl[i]];
		if (theSlot->bits == NULL) { // create bitset
			x = theSlot->h_idx;
			y = theSlot->w_idx;
			length = theSlot->length;
			direction = theSlot->direction;
			theSlot->bits = FindWords(x, y, length, direction, dict);
		} // create bitset
		if (theSlot->bits->count < 0) { // count bits
			BitSetCount(theSlot->bits);
		}
		//fprintf(stderr, "index %d num %d\n", wtbl[i], numbits);
		if(theSlot->bits->count < result)
		{
			//fprintf(stderr, "Replacing result %d with numbits %d\n", 
			//        result, numbits);
			result = theSlot->bits->count;
			resultidx = i;
		}
	}
	// If all words have at least LIMIT solutions each, then return the current
	// index.
	if (result == LIMIT)
		return index;
	// return the index of the word with the least number of solutions.
	return resultidx;
} // FindNextWord

/*
 * Function: overlaps
 * Purpose: see if two slots overlap
 * returns: boolean
 */
int overlaps(slot_list_t *first, slot_list_t *second) {
	if (second->direction == first->direction) {
		// fprintf(stderr, "dir %d, (%d,%d)[%d] does not intersect (%d,%d)[%d]\n",
		// 	first->direction,
		// 	first->h_idx, first->w_idx, first->length,
		// 	second->h_idx, second->w_idx, second->length);
		return(0);
	}
	if (first->direction == ACROSS) {
		// various situations in which first does not intersect
		// second
		if ((second->w_idx < first->w_idx) ||
		(second->w_idx >= first->w_idx + first->length) ||
		(second->h_idx > first->h_idx) ||
		(second->h_idx + second->length <= first->h_idx)) {
			// fprintf(stderr, "A(%d,%d)[%d] does not intersect D(%d,%d)[%d]\n",
			// 	first->h_idx, first->w_idx, first->length,
			// 	second->h_idx, second->w_idx, second->length);
			return(0);
		}
		return(1); // overlap
	} else { // DOWN
		if ((second->h_idx < first->h_idx) ||
		second->h_idx >= first->h_idx + first->length ||
		second->w_idx > first->w_idx ||
		second->w_idx + second->length <= first->w_idx) {
			// fprintf(stderr, "D(%d,%d)[%d] does not intersect A(%d,%d)[%d]\n",
			// 	first->h_idx, first->w_idx, first->length,
			// 	second->h_idx, second->w_idx, second->length);
			return(0);
		}
		return(1); // overlap
	} // DOWN
} // overlaps

void verifyCounts(int *wtbl) { // for debugging: check all counts
	int index;
	for (index = 0; index < NUM_WORDS; index += 1) {
		if (slotlist[wtbl[index]].bits) {
			BitSetCount(slotlist[wtbl[index]].bits);
		}
	} // one bitset to verify
} // verifyCounts

/*
 * Function: backjumpCMD
 * Purpose: starts the backjumping algorithm to solve the puzzle.
 * I/O: In  - Standard Tk arguments
 *      Out - Standard Tk return codes
 * Returns: TCL_OK/TCL_ERROR
 * Functions called: backjump()
*/
int backjumpCMD(ClientData clientdata, Tcl_Interp *interp, int argc,
                 char *argv[])
{
	int selectedword;
	int slotIndex; // which slot (0 ..) we are working on
	int length;
	int wtbl[NUM_WORDS];
	int tblcount;
	int index, tempindex;
	// struct timeval starttime, stoptime;
	struct rusage startUsage, stopUsage;
	int numTries[500]; // for each level, how many times we tried a word
 
	NumBacktracks = 0;
	// gettimeofday(&starttime, NULL);
	getrusage(RUSAGE_SELF, &startUsage);
	bzero(numTries, sizeof(int)*500);

	for(tblcount=0;tblcount<NUM_WORDS;tblcount++)
	{
		wtbl[tblcount] = tblcount;
	}
	
	slotIndex = 0;

	// Find the first slot to solve
	index = FindNextWord(slotlist, wtbl, 0, dict);
	// swap next slot into wtbl[slotIndex]
	tempindex = wtbl[slotIndex]; 
	wtbl[0] = wtbl[index];
	wtbl[index] = tempindex;
	// Main routine
	while ((slotIndex >= 0)&&(slotIndex<NUM_WORDS))
	{
		// verifyCounts(wtbl); // debugging
		slot_list_t *mySlot = &slotlist[wtbl[slotIndex]];
		// Count the number of words left for this index
	  
		//PrintAnswer();
		//sleep(1);
		//fprintf(stdout, "\n");
		length = mySlot->length;
		BitSetCount(mySlot->bits);
		// fprintf(stderr, "idx=%d wtbl=%d remaining=%d\n", slotIndex, wtbl[slotIndex], mySlot->bits->count);
     trial:
		if (mySlot->bits->count == 0) { // no candidates left for current slot
			// fprintf(stderr, "that was the last one\n");
			NumBacktracks += 1;
			FreeBitSet(mySlot->bits);
			mySlot->bits = NULL;
			slotIndex--;
			mySlot = &slotlist[wtbl[slotIndex]];
			if (slotIndex>=0) { // there is a word to remove
				// fprintf(stdout, "3: Removing %d\n", wtbl[slotIndex]);
				RemoveWordGrid(mySlot->h_idx, mySlot->w_idx,
					mySlot->direction, mySlot->length);
				dict[mySlot->length-1].words[mySlot->wordindex].isTaken=0;
				mySlot->wordindex = -1;
				// FreeBitSet(mySlot->bits);
				// mySlot->bits = NULL;
			} // there is a word to remove
			continue;
		} // no candidates left for current slot
		else
		{  // we still have candidates for the current slot
			// Take one out and check to see if it's being used. 
		again: // we'll clean this up later.  We jump here if forward checking denies.
			// verifyCounts(wtbl); // debugging
			selectedword = BitSetExtract(mySlot->bits);
			// verifyCounts(wtbl); // debugging
			if (selectedword == -1) { // rare case of miscalculation
				mySlot->bits->count = 0;
				continue;
			}	
			// we can't trust any precomputed bits for unfilled slots
			int index;
			for (index = slotIndex+1; index < NUM_WORDS; index += 1) {
				slot_list_t *theSlot = &slotlist[wtbl[index]];
				if (theSlot->bits) {
					FreeBitSet(theSlot->bits);
					theSlot->bits = 0;
				}
			}
			numTries[slotIndex] += 1;
			if (dict[length-1].words[selectedword].isTaken)
			{
				mySlot->wordindex = -1;
				continue;
			}
			else
			{ // word is still available to be used
				//fprintf(stdout, "Inserting %d length %d\n", wtbl[slotIndex], 
				//        length);
				if (0) { // debugging
					wide_t *widePtr = dict[length-1].words[selectedword].string;
					char word[100];
					int index;
					for (index = 0; index < dict[length-1].words[selectedword].ucslength; index += 1) {
						word[index] = widePtr[index];
					}
					word[index] = 0;
					fprintf(stderr, "selectedword %s (idx %d; %d left)\n",
						word, slotIndex, mySlot->bits->count);
				} // debugging
				slot_list_t *wordPtr = &slotlist[wtbl[slotIndex]];
				InsertWordGrid(
					dict[length-1].words[selectedword].string,
					dict[length-1].words[selectedword].ucslength,
					wordPtr->h_idx,
					wordPtr->w_idx,
					wordPtr->direction,
					wordPtr->length);
				dict[length-1].words[selectedword].isTaken = 1;
				wordPtr->wordindex = selectedword;
				// forward checking: does the new word cause problems later?
				int futureSlotIndex;
				for (futureSlotIndex = slotIndex+1; futureSlotIndex < NUM_WORDS; futureSlotIndex += 1) {
					if (overlaps(mySlot, &slotlist[wtbl[futureSlotIndex]])) {
						// fprintf(stderr, "?");
						if (!couldFill(&slotlist[wtbl[futureSlotIndex]])) {
							// fprintf(stderr, "forward checking forbids, level %d\n", slotIndex);
							RemoveWordGrid(wordPtr->h_idx, wordPtr->w_idx,
								wordPtr->direction, wordPtr->length);
							dict[wordPtr->length-1].words[selectedword].isTaken=0;
							wordPtr->wordindex = -1;
							if (mySlot->bits->count > 0) { // no more possibilities
								goto again;
							} // no more possibilities for this slot
							// fprintf(stderr, "No candidates left for idx=%d\n", slotIndex);
							goto trial; // this word is no good; select another
						}
					} // overlaps
				} // each future slot
				// fprintf(stderr, "\n");	
			} // word is still available to be used
			if(slotIndex==NUM_WORDS-1)
			{ // no more slots to fill
				//LogAnswer();
				//PrintAnswer();
				//fprintf(stdout, "\n");
				//fprintf(stdout, "Done.\n");
				// gettimeofday(&stoptime, NULL);
				getrusage(RUSAGE_SELF, &stopUsage);
				timeval_diff(startUsage.ru_utime, stopUsage.ru_utime,
					&TotalTime);
				WriteLog("Time: %4.2f Backtracks: %ld\n", TotalTime,
					NumBacktracks);
				for(index=0;index<NUM_WORDS;index++)
				{
					mySlot = &slotlist[wtbl[index]];
					FreeBitSet(mySlot->bits);
					mySlot->bits = 0;
					WriteLog("index %d: %d trials\n", index, numTries[index]);
				}
				FreeDict(dict);
				int clearcnt;
				for(clearcnt=0;clearcnt<NUM_WORDS;clearcnt++)
				{
					selectedword = slotlist[wtbl[clearcnt]].wordindex;
					slotlist[wtbl[clearcnt]].wordindex = -1;
					length = slotlist[wtbl[clearcnt]].length;
					dict[length-1].words[selectedword].isTaken = 0;
				}
				for(clearcnt=0;clearcnt<MAXGRIDSIZE;clearcnt++)
				{
					lengths[clearcnt]=0;
				}
				return TCL_OK;
			} // no more slots to fill
			else
			{ // more slots to fill
				// fill the next slot (deeper into backtrack)
				//LogAnswer();
				//PrintAnswer();
				//fprintf(stdout, "%d words solved.\n", slotIndex);
				slotIndex++;
				// We find the next word to solve.
				index = FindNextWord(slotlist, wtbl, slotIndex, dict);
				tempindex = wtbl[slotIndex]; 
				wtbl[slotIndex] = wtbl[index];
				wtbl[index] = tempindex;
				// Get the list of possible words for this index.
				mySlot = &slotlist[wtbl[slotIndex]];
				GetWorkingSet(slotlist, wtbl, slotIndex, dict);
			} // fill the next slot (deeper into backtrack)
		} // while we have candidates for the current slot
	} // while there is a slot in the crossword to fill

	NOSOLUTION = 1;
	return TCL_OK;
} // backjumpCMD
