#ifndef __SOCKET_LIB_H
#define __SOCKET_LIB_H


 
/** \addtogroup socket_lib */
/** @{*/
 
/*! \file socket_lib.h
*   \brief This is the header for the Socket Library
*   \author Daniel R. Warren
*   \version 1.0
*   \date    November 2004
*/


#include <stdlib.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <signal.h>
#include <string.h>
#include <arpa/inet.h>
#include <errno.h>
#include "log_lib.h"

#define SOCKET_LIB_LOG_BUFF_SIZE 200

#define SOCKET_LIB_NAME "socket_lib"
#define SOCKET_LIB_LOG_FILE "socket_lib.log"

#define SOCKET_LOGMESSAGE(string, location) logmessage(string, location ,\
	       	SOCKET_LIB_NAME, SOCKET_LIB_LOG_FILE, __FILE__, __LINE__,\
	       	__FUNCTION__); 


/** \brief This function calls the accept function with additional logging
 *
 * A programmer can use this function to accept client connections on
 * the new server port. This function uses the same algorithm as the accept function call with
 * additional logging on the connecting client.  
 * 
 * @param sock The socket over which to accept a connection.
 * 
 * @return If the socket is accepted successfully, the function returns the new socket
 * descriptor; otherwise it returns -1.
 * 
**/

int accept_connect(int sock);



/** \brief Allows a client to establish a TCP connection with the supplied
 *         IP address and port.
 * 
 * This function can be used to establish a TCP connection to a server
 * listening on a specified port.
 * 
 * @param ipaddr The IP address and port number of the server with which to
 *               establish a connection.
 * @param port The port of the server with which to establish a connection.
 * 
 * @return If the connection is established, the function returns the new
 *         socket descriptor; otherwise it returns -1.
 *
**/

int create_connect(struct in_addr *ipaddr, short port);



/** \brief This function creates a server socket by binding and listening on the
 *         supplied port.
 * 
 * One of the more common network tasks is creating a server socket that binds
 * and listens for activity on a specified port.  This function accomplishes
 * this task for the user.  By default this function binds the port on all
 * available network interfaces.
 * 
 * @param port The port to bind and listen on.
 * @param queue_len The maximum number of client connections queued while
 *                  waiting to be accepted.
 *
 * @return If the function creates the socket successfully, the function returns the
 * new socket descriptor; otherwise it returns -1.
 *
**/

int create_serv_socket(int port, int queue_len);



/** \brief Allows data to be sent safely over the network without fear of
 *         the program terminating.
 * 
 * In order to send data over an established TCP connection, it is important
 * that the data be transfered in a manner that is safe for the calling process
 * and ensures the transfer occurred.  When a programmer uses the normal network
 * send function, the calling process is terminated if the process tries to
 * send data over a socket that is closed.  This function provides protection
 * from this by blocking signals and creating timeouts.
 *
 * @param sock The socket descriptor to send or receive data over.
 * @param send_buff A previously allocated buffer of length send_len that is
 *                  filled with data to send over the socket.
 * @param send_len The length of send_buff.
 * @param timeout The maximum time in seconds that the send function
 *                will wait until data is sent.
 *
 * @return This function returns -1 on failure or the number of bytes sent.
 *  
 * @see safe_sock_recv
**/

int safe_sock_send(int sock, void *send_buff, unsigned int send_len,
                                             unsigned int timeout);




/** \brief Allows data to be received over the network without having to
 *         manually check that all the data was received.
 * 
 * In order to receive data over an established TCP connection, it is 
 * important that the data be transfered in a manner that is safe for the
 * calling process and ensures the transfer occurred.  The normal network
 * receive function causes the user to make multiple function calls or wait
 * indefinitely.  The following functions provide protection from these problems
 * by blocking signals and creating timeouts.
 *
 * @param sock The socket descriptor receive data over.
 * @param recv_buff A previously allocated buffer of length recv_len that is
 *                  filled on success with data to received over the socket.
 * @param recv_len The length of recv_buff.
 * @param timeout The maximum time in seconds that the receive function
 *                will wait until data is received.
 *
 * @return This function returns -1 on failure or the number of bytes received.
 *  
 * @see safe_sock_send
**/

int safe_sock_recv(int sock,void *recv_buff, unsigned int recv_len,
                                             unsigned int timeout);

/** @}*/

#endif
