/*! \file socket_lib.c
 *  \brief This is the implementation of the Socket Library
 *  \author Daniel R. Warren
 *  \version 1.0
 *  \date    November 2004
 *  \ingroup socket_lib
*/


 /** \defgroup socket_lib The Socket Library
 *   \ingroup helper_components
 *   \brief Library that provides common functionality used in TCP applications.
 * 
 * Many of today's applications require sending and receiving messages over a
 * network.  In order to make this task more programmer-friendly, I developed a
 * component that provides an interface for the most common network functions.
 * The component sends and receives data over network sockets using TCP in the
 * transport layer.  TCP is commonly used among network applications and
 * provides the reliability and speed required for most applications.
 *
 *
 */
 
/** \addtogroup socket_lib */
/** @{*/




#include "socket_lib.h"

static void catch_alarm();

static void catch_alarm() {
   alarm(0);
}




int accept_connect(int sock) {
   int ret_sock;
   int socklength;
   struct sockaddr_in *recv_addr;
   char log_buffer[SOCKET_LIB_LOG_BUFF_SIZE];

   memset(log_buffer, 0, SOCKET_LIB_LOG_BUFF_SIZE);
   socklength = sizeof(struct sockaddr_in);

   recv_addr = (struct sockaddr_in *) malloc(socklength);
   if(recv_addr==NULL) {
      SOCKET_LOGMESSAGE("malloc() failed",LOGSTDERR);
      return -1;
   }

   if((ret_sock=accept(sock, (struct sockaddr *)recv_addr, &socklength))<0){
      SOCKET_LOGMESSAGE("accept() failed", LOGSTDERR);
      free(recv_addr);
      return -1;
   }

   sprintf(log_buffer, "Handling client from %s",
		                 inet_ntoa(recv_addr->sin_addr));
   SOCKET_LOGMESSAGE(log_buffer, LOGSTDOUT);

   free(recv_addr);
   return ret_sock;
}











int create_connect(struct in_addr *ipaddr, short port) {
   int newsock;
   struct sockaddr_in send_addr;

   memset(&send_addr, 0, sizeof(send_addr));

   send_addr.sin_family = AF_INET;
   memcpy(&send_addr.sin_addr, ipaddr, sizeof(struct in_addr));

   send_addr.sin_port = htons(port);

   if((newsock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0) {
      SOCKET_LOGMESSAGE("Could not create socket.", LOGSTDERR);
      return -1;
   }

   if(connect(newsock,(struct sockaddr *) &send_addr,
              sizeof(struct sockaddr))<0) {
      close(newsock);
      return -1;
   }

   return newsock;
}









int create_serv_socket(int port, int queue_len) {
   int sock;
   struct sockaddr_in serv_addr;
   char log_buffer[SOCKET_LIB_LOG_BUFF_SIZE];

   memset(log_buffer, 0, SOCKET_LIB_LOG_BUFF_SIZE);
   memset(&serv_addr, 0, sizeof(serv_addr));

   if((sock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0) {
      SOCKET_LOGMESSAGE("socket() failed", LOGSTDERR);
      return -1;
   }

   serv_addr.sin_family = AF_INET;
   serv_addr.sin_addr.s_addr = htonl(INADDR_ANY);
   serv_addr.sin_port = htons(port);

   if(bind(sock,(struct sockaddr *) &serv_addr,sizeof(serv_addr))<0){
      SOCKET_LOGMESSAGE("bind() failed", LOGSTDERR);
      return -1;
   }

   if(listen(sock, queue_len) <0) {
      SOCKET_LOGMESSAGE("listen() failed", LOGSTDERR);
      return -1;
   }
   
   sprintf(log_buffer, "Listening on TCP Port %d...", port);
   SOCKET_LOGMESSAGE(log_buffer, LOGSTDOUT);


   return sock;
}








int safe_sock_send(int sock, void *send_buff, unsigned int send_len,
                                             unsigned int timeout){

   unsigned int justsent;
   unsigned int totalsent;
   struct sigaction catchalarm;

   catchalarm.sa_handler = catch_alarm;
   if(sigfillset(&catchalarm.sa_mask) < 0) {
      SOCKET_LOGMESSAGE("sigfillset() failed", LOGSTDERR);
      return -1;
   }

   catchalarm.sa_flags = 0;
   if(sigaction(SIGALRM, &catchalarm, 0) < 0) {
      SOCKET_LOGMESSAGE("sigaction() failed.", LOGSTDERR);
      return -1;
   }

   totalsent=0;
   signal(SIGPIPE,SIG_IGN);
   alarm(timeout);
   while(totalsent < send_len){
      if((justsent=send(sock,send_buff+totalsent,send_len-totalsent,0))!=
		                          send_len-totalsent) {
         if(errno==EINTR) {
            SOCKET_LOGMESSAGE("send() timed out", LOGSTDERR);
            signal(SIGPIPE,SIG_DFL);
            alarm(0);
            return -1;
         }
         else {
            SOCKET_LOGMESSAGE("send() failed", LOGSTDERR);
            signal(SIGPIPE,SIG_DFL);
            alarm(0);
            return -1;
         }
      }
      else{
         totalsent+=justsent;
      }
   }

   signal(SIGPIPE,SIG_DFL);
   alarm(0);

   return send_len;
}







int safe_sock_recv(int sock,void *recv_buff, unsigned int recv_len,
                                             unsigned int timeout){
   unsigned int justrecv;
   unsigned int totalrecv;
   struct sigaction catchalarm;

   catchalarm.sa_handler = catch_alarm;
   if(sigfillset(&catchalarm.sa_mask) < 0) {
      SOCKET_LOGMESSAGE("sigfillset() failed", LOGSTDERR);
      return -1;
   }

   catchalarm.sa_flags = 0;
   if(sigaction(SIGALRM, &catchalarm, 0) < 0) {
      SOCKET_LOGMESSAGE("sigaction() failed", LOGSTDERR);
      return -1;
   }

   totalrecv=0;
   alarm(timeout);

   while(totalrecv<recv_len) {
      if((justrecv=recv(sock,recv_buff+totalrecv,recv_len-totalrecv,0))<0) {
         if(errno==EINTR) {
            SOCKET_LOGMESSAGE("recv() timed out", LOGSTDERR);
            alarm(0);
            return -1;
         }
         else {
            SOCKET_LOGMESSAGE("recv() failed", LOGSTDERR);
            alarm(0);
            return -1;
         }
      }
      else {
         if(justrecv==0) {
            SOCKET_LOGMESSAGE("recv() returned with socket closed", LOGSTDERR);
            alarm(0);
            return -1;
         }
         else totalrecv+=justrecv;
      }
   }

   alarm(0);

   return recv_len;
}

/** @}*/

