
/** \mainpage Alsad Overview
 *
 * \section abstract Abstract
 *
 * This paper describes my implementation of alsa_client and alsad, client and
 * server applications written for the GNU/Linux operating system, which give
 * users the ability to create audio streams with user-defined sources and
 * sinks.  The current implementation allows the user to create sources and
 * sinks that are sound-card hardware interfaces or clients running alsad's 
 * custom network protocol.  Alsad transfers sound data from a source to all 
 * of the existing sinks.  Alsa_client and  alsad  transfer uncompressed audio
 * data over the network.  As a result, these applications are most suited for
 * use on local area networks.

   Users of alsad and alsa_client can create an audio stream where the source 
   is a file or a sound-card hardware interface, optionally stream the sound 
   data to multiple alsads with additional sinks, and ultimately write the 
   streamed audio data to a sound-card hardware interface or a file.

 * 
 * \section overview Overview
 * 
 * Alsad allows connecting clients to add and delete streams, sources,
 * and sinks.  An alsad stream, known hereafter as a stream, is a
 * chronologically ordered flow of sound data with a unique identifier,
 * text description, and properties that define the type of sound data that
 * flows through the stream.  A stream may have a source and must have one or
 * more sinks.  Alsad places data received from a stream's source into a
 * shared buffer for sinks to consume.  As a result, all sinks consume the same
 * data, but may be reading from different parts of the shared buffer.  A
 * sink's position in the buffer depends upon the rate at which its destination
 * asks for data.    

   When a user first adds a stream, it has no source and a sink that alsad
   automatically creates to discard the stream's data.  Once a user creates a
   stream, he/she can then request that alsad add a source or additional sinks
   to this stream.  If the user chooses to add a source first, the automatically
   generated sink removes the sources data as soon as it arrives.  This action
   continues until the user adds a sink.  If the user adds a sink to a stream
   before they add a source, the sink waits until the stream contains data to
   read.  Once data is available,  the sink reads from the buffer and sends the
   data to its defined destination.  If a user requests that alsad remove a
   source, any data received by alsad is left for connected sinks to consume. 

   Sources and sinks for a stream can be sound-card hardware interfaces or
   clients running alsad's custom network protocol.  Sound-card hardware
   interfaces are the hardware input and output devices located on the sound
   card.  They connect the sound card with external components that attach to 
   the card.  Microphones, headphones and low-level input and output devices 
   (both in digital and analog format) use sound-card hardware interfaces to 
   connect with the sound card.  Alsad uses the Advanced Linux Sound 
   Architecture (ALSA) libraries to configure and interact with the sound card 
   when a stream has a source or a sink that is a sound-card hardware interface.

   A user has three options when determining the rate sound data should flow 
   though a stream.  In the first option, alsad calculates the rate of flow 
   based on user-defined stream properties that describe the real-time rate of 
   the stream data.  Alsad only allows sources to send data at this rate.  
   All sinks are in turn forced to read at this rate or receive an underrun.  
   With the second option, alsad allows a source to send data as fast as the 
   slowest sink can consume it (if there are no sinks the source is allowed 
   unlimited write permission).  This strategy prevents any sinks from receiving
   an underrun.  In the final option, alsad combines the last two strategies by
   choosing the fastest of: the real-time rate of option one and the
   slowest-sink rate of option two.  This strategy gives a underrun to clients
   reading slower than the real-time rate and allows sinks to consume faster
   than real time if there are no real-time sinks.

   Clients connected to alsad can also request stored information about
   existing streams, sources, and sinks.  The client can obtain this 
   information by making two different requests.  The first request provides 
   clients with a list of general information about defined streams on alsad.  
   This information includes the stream's unique identifier, the textual 
   description of the stream, and properties that define the sound data that 
   passes through the stream.  The second request allows clients that specify 
   an existing stream to receive all relevant source and sink information.  
   Alsad sends the unique identifier of the sources and sinks for the requested 
   stream.  If the source or sink is a sound-card hardware interface, alsad 
   also includes settings to configure the hardware.  In addition to this 
   information, alsad also sends address information for every connected client.

   When a client adds a source or a sink that is not a sound-card hardware 
   interface, it must also send a message containing the hostname of the 
   machine on which it is executing.  If the connecting client is an alsad, it 
   sends the TCP port on which it accepts connections.  Otherwise, the client 
   sends 0 as the TCP port.  Storing this information allows intelligent 
   clients to query alsads, using the queries described in the last paragraph, 
   to find alternative locations for a desired stream.  Finding an alternative 
   location might be important in situations where clients are consuming a 
   large percentage of an alsad's network bandwidth.  This lack of bandwidth 
   may prevent clients from creating a sustainable sink on an alsad.  If the 
   desired stream has an existing sink that is an alsad, an intelligent client 
   could try to connect with this alternative.  If the alternative alsad is on 
   a network segment with enough bandwidth to handle transporting the desired 
   stream, the client can create a sink for this stream.

 * 
 */


/** \defgroup the_applications The Applications
 *  \brief The applications created for this project.
 */
 
 
/** \defgroup application_components Application-Specific Components
 *  \brief Components meant to be used only with these applications.
 */

/** \defgroup helper_components Helper Components
 *  \brief Components that could be used outside of alsad to 
 *          perform common tasks.
 */
 
 
 /** \defgroup alsad The ALSA Sound Server
 *   \ingroup the_applications
 *   \brief The Distributed Audio Server Application
 **/
 
 /*! \file main.c
 *  \brief This is the main file for alsad
 *  \author Daniel R. Warren
 *  \version 1.0
 *  \date    November 2004
 *  \ingroup alsad
*/

 
 
/** \addtogroup alsad */
/** @{*/
 
#include <stdlib.h>
#include <stdio.h>
#include <sys/file.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/time.h>
#include <string.h>
#include <ctype.h>
#include <netdb.h>
#include <sys/time.h>
#include <pthread.h>
#include <linux/un.h>
#include <linux/stddef.h>
#include <getopt.h>



#include "auth_lib.h"
#include "alsad_defs.h"
#include "alsad_net_lib.h"
#include "socket_lib.h"
#include "log_lib.h"
#include "alsad_stream_list.h"
#include "alsad_data_pipe_list.h"
#include "alsad_shared_lib.h"
#include "circ_buff.h"
#include "config_file.h"

#define ALSAD_CONFIG_FILE "/etc/alsad/alsad.conf"
#define ALSAD_LOG_FILE "/etc/alsad/alsad.log"
#define ALSAD_MAX_NET_QUEUE 30
#define ALSAD_SELECT_TIMEOUT 4
#define ALSAD_THREAD_CHANNEL_WAIT 4 
#define ALSAD_THREAD_SHUTDOWN_TIME 6
#define ALSAD_BUFF_SIZE_SCALE 128
#define ALSAD_DEF_CIRC_TIMEOUT 1
#define ALSAD_MAX_COMPRESS 10L


//Make this an option in the config file
#define ALSAD_DEFAULT_AUDIO_BUF_SIZE 1024


#define ALSAD_PROC_NAME "alsad"
#define LOGMESSAGE(string, location) logmessage(string, location ,\
                ALSAD_PROC_NAME, ALSAD_LOG_FILE, __FILE__, __LINE__,\
                __FUNCTION__);

/*  Global Variables  */
char buffer[50];
int server_port;
int alsad_shutdown;
char *config_file;

struct alsad_handle_client_args{
   int sock;
   alsad_stream_list_t *list;
};

typedef struct alsad_handle_client_args alsad_handle_client_args_t;


struct alsad_drain_buff_args{
   alsad_stream_list_t *list;
   alsad_stream_node_t *stream;
};

typedef struct alsad_drain_buff_args alsad_drain_buff_args_t;



/* Function Prototypes */


void *alsad_handle_client(void *arguments);
void *alsad_drain_buffer(void *arguments);
/*int alsad_handle_create_chan(int sock, snd_pcm_t **handle, char **audiobuf,
                                 alsad_stream_list_t *head_of_list,
                                char *priv_key_file, char *userid,
                                int from_chan, int play);
*/
int alsad_handle_add_stream(int sock, alsad_stream_list_t *head_of_list,
                                     alsad_config_keys_t *load_keys,
                                     alsad_control_t *return_ctrl);
                                     
int alsad_handle_del_stream(int sock, alsad_stream_list_t *head_of_list,
                                     alsad_control_t *return_ctrl);


int alsad_handle_add_src(int *sock, alsad_stream_list_t *head_of_list,
                         alsad_control_t *rcvd_control,
                         alsad_config_keys_t *load_keys,
                         alsad_control_t *return_ctrl);
                         
int alsad_handle_del_src(int sock, alsad_stream_list_t *head_of_list,
                                   alsad_control_t *rcvd_control,
                                   alsad_control_t *return_ctrl);

int alsad_handle_add_sink(int *sock, alsad_stream_list_t *head_of_list,
                          alsad_control_t *rcvd_control,
                          alsad_config_keys_t *load_keys,
                          alsad_control_t *return_ctrl);
                          
int alsad_handle_del_sink(int sock, alsad_stream_list_t *head_of_list,
                                   alsad_control_t *rcvd_control,
                                   alsad_control_t *return_ctrl);


int alsad_handle_list_stream(int sock, alsad_stream_list_t *head_of_list,
                                   alsad_control_t *return_ctrl);

int alsad_handle_list_pipe(int sock, alsad_stream_list_t *head_of_list,
                                   alsad_control_t *rcvd_control,
                                   alsad_control_t *return_ctrl);

int alsad_is_server(alsad_connect_addr_t *connection);

int alsad_data_pipe_compare(void *identifier, void *data_pipe);

int alsad_process_data_pipe(void *data_pipe, void *args);

static void catch_signal(int signal);





int main(int argc, char **argv) {
   
   int server_sock_desc;
   int client_sock_desc;
   int ret_val;
   fd_set listen_fd;
   struct timeval timeout;
   pthread_t thread_ident;
   alsad_handle_client_args_t *funct_args;
   alsad_stream_list_t head_of_channel_list;
   alsad_stream_list_init(&head_of_channel_list, 0);
   alsad_shutdown=0;
   server_port=0;
   config_file=NULL;
   
   int ind_opts;
   int opt;
   char *optstring="f:p:";
   static struct option longopts[] = {
      {"server_port",1,0,'p'},
      {"config_file",1,0,'f'},
      {0,0,0,0}
   };
   

   while((opt = getopt_long(argc,argv, optstring, longopts, &ind_opts))!=-1){
      switch(opt){
      case 'f':
         config_file = strdup(optarg);
         break;
      case 'p':
         server_port=atoi(optarg);
         break;
      default:
         fprintf(stderr,"Invalid option exiting....\n");
         exit(1);
      };
   }
   if(server_port==0){
      LOGMESSAGE("Usage: alsad -p<localport> -f<config_file>", LOGSTDERR);
      exit(0);
   }
         
   signal(SIGINT, catch_signal);
   signal(SIGTERM, catch_signal);
   signal(SIGABRT, catch_signal);

   
   /* set up a socket and bind on port described in the alsad.config
        file then listen i.e create_serv_socket*/
   server_sock_desc = create_serv_socket(server_port, ALSAD_MAX_NET_QUEUE);
   if(server_sock_desc < 0){
      LOGMESSAGE("Could not create server socket... exiting\n",LOGSTDERR);
      exit(1);
   }
   

   while(!alsad_shutdown){
      ret_val=0;
      while(ret_val==0 && !alsad_shutdown){
         FD_ZERO(&listen_fd);
         FD_SET(server_sock_desc, &listen_fd);
         timeout.tv_sec = ALSAD_SELECT_TIMEOUT; 
         timeout.tv_usec = 0;
         ret_val=select(server_sock_desc+1, &listen_fd, NULL, NULL, &timeout);
         if(ret_val<0){
            alsad_shutdown=1;
         }
      }
      if(alsad_shutdown)
         continue;
      client_sock_desc = accept_connect(server_sock_desc);
      if(client_sock_desc<0)
         continue; 
      
      funct_args=(alsad_handle_client_args_t *)malloc(sizeof(alsad_handle_client_args_t));
      if(funct_args==NULL){
          LOGMESSAGE("Could not allocate memory for funct_args",LOGSTDERR);
          return -1;
      }
      funct_args->sock=client_sock_desc;
      funct_args->list=&head_of_channel_list;
      
      ret_val=pthread_create(&thread_ident,NULL,alsad_handle_client,
                               (void *)funct_args);
      if(ret_val!=0){
          LOGMESSAGE("Could not create a thread",LOGSTDERR);
          return -1;
      }

   }
   head_of_channel_list.shutdown=1;
   sleep(ALSAD_THREAD_SHUTDOWN_TIME);
   return 0;
}  // End of main()






void *alsad_handle_client(void *arguments){
   
   int retval, client_sock_desc;
   alsad_control_t rcvd_control;
   alsad_stream_list_t *head_of_list;
   alsad_config_keys_t load_keys;
   int session_terminated = 0;
   alsad_control_t return_ctrl;
   char print_buff[80];
   pthread_detach(pthread_self());

   client_sock_desc=((alsad_handle_client_args_t *)arguments)->sock;
   head_of_list=((alsad_handle_client_args_t *)arguments)->list;

   free(arguments);

   if(config_file==NULL){
      retval=alsad_config_key_load(ALSAD_CONFIG_FILE,&load_keys);
      if(retval<0){
         sprintf(print_buff, 
         "Could not find the default configuration file: %s\n",
         ALSAD_CONFIG_FILE);
         LOGMESSAGE(&print_buff[0],LOGSTDERR);
         close(client_sock_desc);
         return NULL;
      }
   }
   else{
      retval=alsad_config_key_load(config_file,&load_keys);
      if(retval<0){
         sprintf(print_buff, 
         "Could not find the user defined configuration file: %s\n",
         config_file);
         LOGMESSAGE(&print_buff[0],LOGSTDERR);
         close(client_sock_desc);
         return NULL;
      }
   }
/**\todo make authenticating an option set from the config file
 */
   retval=auth_server(client_sock_desc,load_keys.config_pub_key_dir);
   if(retval>0){
      LOGMESSAGE("Failed to authenticate user", LOGSTDERR);
      session_terminated=1;
   }

   while(!session_terminated){

      /*receive one byte that decribes the incoming protocol that follows.
      * The initial byte definitions are in alsad_net_lib.h*/
      if((retval = alsad_recv_control(client_sock_desc, &rcvd_control)) <= 0){
         LOGMESSAGE("Failed to recv control message from client", LOGSTDERR);
         session_terminated=1;
         close(client_sock_desc);
         continue;
      }

      /*act out the request of the communication*/
      if((rcvd_control.ctrl & ALSAD_RQST_ADD_STREAM)==
                                              ALSAD_RQST_ADD_STREAM){
         LOGMESSAGE("Starting handle add chan", LOGSTDOUT);
         if((retval=alsad_handle_add_stream(client_sock_desc,
                                                   head_of_list,
                                                   &load_keys,
                                                   &return_ctrl))<0){
            session_terminated=1;
            continue;
         }
         if((retval=alsad_send_control(client_sock_desc, &return_ctrl)) <= 0){
            LOGMESSAGE("Failed to send control message to client", LOGSTDERR);
            session_terminated=1;
            close(client_sock_desc);
         }
         continue;
      }
      
      if((rcvd_control.ctrl & ALSAD_RQST_DEL_STREAM) == ALSAD_RQST_DEL_STREAM){
         if((retval=alsad_handle_del_stream(client_sock_desc,
                                          head_of_list,&return_ctrl))<0){
            session_terminated=1;
            continue;
         }
         if((retval=alsad_send_control(client_sock_desc, &return_ctrl)) <= 0){
            LOGMESSAGE("Failed to send control message to client", LOGSTDERR);
            session_terminated=1;
            close(client_sock_desc);
         }
         continue;
      }


      if((rcvd_control.ctrl & ALSAD_RQST_ADD_SRC) == ALSAD_RQST_ADD_SRC){
         if((retval=alsad_handle_add_src(&client_sock_desc,head_of_list,
                                         &rcvd_control, &load_keys,
                                         &return_ctrl))<0){
            session_terminated=1;
            continue;
         }
         if((retval=alsad_send_control(client_sock_desc, &return_ctrl)) <= 0){
            LOGMESSAGE("Failed to send control message to client", LOGSTDERR);
            session_terminated=1;
            close(client_sock_desc);
         }
         continue;
      }

      if((rcvd_control.ctrl & ALSAD_RQST_DEL_SRC) == ALSAD_RQST_DEL_SRC){
         if((retval=alsad_handle_del_src(client_sock_desc,
                                         head_of_list,
                                         &rcvd_control,
                                         &return_ctrl))<0){
            session_terminated=1;
            continue;
         }
         if((retval=alsad_send_control(client_sock_desc, &return_ctrl)) <= 0){
            LOGMESSAGE("Failed to send control message to client", LOGSTDERR);
            session_terminated=1;
            close(client_sock_desc);
         }
         continue;
      }

      if((rcvd_control.ctrl & ALSAD_RQST_ADD_SINK) == ALSAD_RQST_ADD_SINK){
      LOGMESSAGE("Going to handle add_sink", LOGSTDOUT);
         if((retval=alsad_handle_add_sink(&client_sock_desc,head_of_list,
                                          &rcvd_control, &load_keys,
                                          &return_ctrl))<0){
            session_terminated=1;
            continue;
         }
         if((retval=alsad_send_control(client_sock_desc, &return_ctrl)) <= 0){
            LOGMESSAGE("Failed to send control message to client", LOGSTDERR);
            session_terminated=1;
            close(client_sock_desc);
         }
         continue;
         
      }

      if((rcvd_control.ctrl & ALSAD_RQST_DEL_SINK) == ALSAD_RQST_DEL_SINK){
         if((retval=alsad_handle_del_sink(client_sock_desc, head_of_list,
                                           &rcvd_control,
                                           &return_ctrl))<0){
            session_terminated=1;
            continue;
         }
         if((retval=alsad_send_control(client_sock_desc, &return_ctrl)) <= 0){
            LOGMESSAGE("Failed to send control message to client", LOGSTDERR);
            session_terminated=1;
            close(client_sock_desc);
         }
         continue;
      }

      if((rcvd_control.ctrl & ALSAD_RQST_LIST_STREAM) 
                                                == ALSAD_RQST_LIST_STREAM){
         if((retval=alsad_handle_list_stream(client_sock_desc,
                                           head_of_list,
                                           &return_ctrl))<0){
            session_terminated=1;
            continue;
         }
         if((retval=alsad_send_control(client_sock_desc, &return_ctrl)) <= 0){
            LOGMESSAGE("Failed to send control message to client", LOGSTDERR);
            session_terminated=1;
            close(client_sock_desc);
         }
         continue;
      }

      if((rcvd_control.ctrl & ALSAD_RQST_LIST_PIPE) == ALSAD_RQST_LIST_PIPE){
         if((retval=alsad_handle_list_pipe(client_sock_desc, head_of_list,
                                           &rcvd_control,
                                           &return_ctrl))<0){
            session_terminated=1;
            continue;
         }
         if((retval=alsad_send_control(client_sock_desc, &return_ctrl)) <= 0){
            LOGMESSAGE("Failed to send control message to client", LOGSTDERR);
            session_terminated=1;
            close(client_sock_desc);
         }
         continue;
      }
      
      if((rcvd_control.ctrl & ALSAD_MSG_CLOSE_CONN) == ALSAD_MSG_CLOSE_CONN){
         LOGMESSAGE("Closing Connection Gracefully", LOGSTDOUT);
         session_terminated=1;
         close(client_sock_desc);
         continue;
      }


      LOGMESSAGE("Invalid Control", LOGSTDOUT);
      session_terminated=1;
      close(client_sock_desc);


   }

   LOGMESSAGE("Handler finished\n",LOGSTDOUT);
   return (NULL);
}





/** Removes data from the streams circ_buff based on the alsad_stream_node_t.
 * 
 * This function is called in its own thread of execution by 
 * alsad_handle_add_stream.  It removes sound data from the streams circ_buff at
 * a rate that is defined by the streams alsad_stream_node_t.
 * 
 * @param arguments This parameter should to be a alsad_drain_buff_args*
 * @see alsad_handle_add_stream()
 * @return There is no return value captured for this function.
 * 
**/
void *alsad_drain_buffer(void *arguments){
   


   alsad_stream_node_t *drain_stream;
   alsad_stream_list_t *drain_list;
   void *drain_audio;
   int retval_int;
   unsigned long period_size_in_bytes, retval;
   unsigned long variable_period_size;
   unsigned long sink_period_size;
   unsigned long max_period_size;
   unsigned long slowest_byte_id;
   unsigned long start_global_id;
   //unsigned long long calc_bytes_per_sec;
   unsigned long long long_size_in_bytes;
   unsigned long long calc_wait_time_sec;
   unsigned long long calc_wait_time_nsec;
   unsigned long long total_bytes_read;
   unsigned long long calc_bytes_read;
   unsigned long long diff_micro;
   unsigned long long defined_bytes_per_sec;


   struct timespec wait_time;
   struct timeval start_time;
   struct timeval diff_time;
   struct timeval curr_time;
   struct timezone timez;

   timez.tz_minuteswest=0;
   total_bytes_read=0;
   pthread_detach(pthread_self());
  
   drain_stream=((alsad_drain_buff_args_t *)arguments)->stream;
   drain_list=((alsad_drain_buff_args_t *)arguments)->list;

   free(arguments);
  
   start_global_id=drain_stream->audio_circ_buff->byte_id;
   //This is the size of chunk to drain out of the circular buffer
   period_size_in_bytes=drain_stream->audio_circ_buff->buff_size*0.05;
   //This is the maximum chunk that can be taken out.
   max_period_size=drain_stream->audio_circ_buff->buff_size*0.25;


   //These two lines make sure that the chunks are divisible by the
   //frame size
   period_size_in_bytes=period_size_in_bytes -
                        (period_size_in_bytes % 
                              drain_stream->stream_props->frame_size);
   max_period_size=max_period_size -
                        (max_period_size % 
                              drain_stream->stream_props->frame_size);



   drain_audio=(void *)malloc(max_period_size);
   if(drain_audio==NULL){
      fprintf(stderr,"Could not allocate memory for drain_audio\n");
      drain_stream->closing=1;
      return NULL;
   }


   long_size_in_bytes=drain_stream->audio_circ_buff->buff_size;
   variable_period_size=period_size_in_bytes;
   
   defined_bytes_per_sec=drain_stream->stream_props->rate*
                              drain_stream->stream_props->frame_size; 
   
   calc_wait_time_sec= (long_size_in_bytes/
                  (drain_stream->stream_props->rate*ALSAD_MAX_COMPRESS*
                   drain_stream->stream_props->frame_size));
   wait_time.tv_sec=(long)calc_wait_time_sec;
   
   calc_wait_time_nsec=(((long_size_in_bytes*1000000000L)/
                  (drain_stream->stream_props->rate*ALSAD_MAX_COMPRESS*
                   drain_stream->stream_props->frame_size))-
                  (calc_wait_time_sec*1000000000L));
   wait_time.tv_nsec=(long)calc_wait_time_nsec;
   
   fprintf(stderr,"rate %d\n",drain_stream->stream_props->rate);
   fprintf(stderr,"frame size %d\n",drain_stream->stream_props->frame_size);
   fprintf(stderr,"period_size_in_bytes %ld\n",period_size_in_bytes);
   fprintf(stderr,"Wait time, Sec: %d nano: %ld\n",
                        (unsigned int) wait_time.tv_sec,wait_time.tv_nsec);
   gettimeofday(&start_time,&timez);

   while(!drain_stream->closing && !drain_list->shutdown){
         //fprintf(stdout,"drain: Bytecount: %ld buff_size: %ld\n",
        //               drain_stream->audio_buff->byte_count,
        //              drain_stream->audio_buff->buff_size);
        

      if((retval_int=circ_buff_wait_for_write(drain_stream->audio_circ_buff,
                                          &wait_time)) < 0){
      }

      
      if(drain_stream->audio_circ_buff->byte_id - start_global_id <=
                                                long_size_in_bytes/2){
         //This is to fix a race condition that may occur when setting
         //start_global_id
         if(drain_stream->audio_circ_buff->byte_count < long_size_in_bytes/2){
            gettimeofday(&start_time,&timez);
            continue;
         }
      }   
      

      if((drain_stream->stream_props->src_permission & ALSAD_SRC_CALC_RATE) == 
                                                      ALSAD_SRC_CALC_RATE){
      
         gettimeofday(&curr_time,&timez);

         if(curr_time.tv_usec < start_time.tv_usec){
             curr_time.tv_sec-=1;
             diff_time.tv_usec=start_time.tv_usec-curr_time.tv_usec;
         }
         else{
            diff_time.tv_usec=curr_time.tv_usec-start_time.tv_usec;
         }

         diff_time.tv_sec=curr_time.tv_sec-start_time.tv_sec;
         //fprintf(stderr,"diff time, Sec: %ld micro: %ld\n",
//                                     diff_time.tv_sec,diff_time.tv_usec);
         diff_micro=(diff_time.tv_sec*1000000)+diff_time.tv_usec; 

         //calc_bytes_per_sec=(unsigned long long)(
         //                   ((total_bytes_read-long_size_in_bytes)*1000000)/
         //                   (diff_micro)); 
         //fprintf(stderr,"Bytes_per_sec=%llu\n",calc_bytes_per_sec);
         
         //calc_nano=(unsigned long long)((long_size_in_bytes*1000000000)/
         //                                  (calc_bytes_per_sec));
         //fprintf(stderr,"calc_nano=%llu\n",calc_nano);



         calc_bytes_read=(defined_bytes_per_sec*diff_micro)/1000000;
         
         if(calc_bytes_read > total_bytes_read)
            variable_period_size=calc_bytes_read-total_bytes_read;
         else
            variable_period_size=0;

         
      }  


      if((drain_stream->stream_props->src_permission & ALSAD_SRC_SINK_RATE) == 
                                                      ALSAD_SRC_SINK_RATE){

          slowest_byte_id=alsad_data_pipe_find_slowest(&(drain_stream->sinks),
                                  drain_stream->audio_circ_buff->byte_id);
          
          if(drain_stream->audio_circ_buff->byte_count >
                           (drain_stream->audio_circ_buff->byte_id -
                                                          slowest_byte_id)){          
             sink_period_size=drain_stream->audio_circ_buff->byte_count - 
                               (drain_stream->audio_circ_buff->byte_id -
                                      slowest_byte_id);
          }
          else{
             sink_period_size=0;
          }
//          fprintf(stderr,"Slow Byte ID: %lu\n",slowest_byte_id);
//          fprintf(stderr,"byte_count - (byte_id - slowest)=sink_period_size\n");
//          fprintf(stderr,"%lu - (%lu - %lu) = %lu\n", drain_stream->audio_circ_buff->byte_count,
//                                              drain_stream->audio_circ_buff->byte_id,
//                                              slowest_byte_id,
//                                              sink_period_size);
      }
      
      //Both sink rate and calc rate
      if((drain_stream->stream_props->src_permission & 
                            (ALSAD_SRC_CALC_RATE | ALSAD_SRC_SINK_RATE)) == 
                            (ALSAD_SRC_CALC_RATE | ALSAD_SRC_SINK_RATE)){
      
         if(sink_period_size > variable_period_size){
            variable_period_size=sink_period_size;
         }
      }
      //not using the calc rate method
      else if(!((drain_stream->stream_props->src_permission &
                                                    ALSAD_SRC_CALC_RATE) == 
                                                    ALSAD_SRC_CALC_RATE)){

         variable_period_size=sink_period_size;
      }

      
      //Make sure that variable_period_size is divisible by the frame_size
      variable_period_size=variable_period_size - 
                              (variable_period_size % 
                              drain_stream->stream_props->frame_size);
                              
      if(variable_period_size > max_period_size){
         variable_period_size=max_period_size;

      }
      
//      fprintf(stderr,"Variable Period Size: %lu\n",variable_period_size);
      
      if(variable_period_size==0)
         continue;
        
      retval=0;    
      while(retval==0 && 
                 !(drain_stream->closing) && !drain_list->shutdown){  
         retval=circ_buff_read(drain_stream->audio_circ_buff,drain_audio,
                                            variable_period_size,
                                            &wait_time);
         if(retval==0){
            gettimeofday(&start_time,&timez);
            total_bytes_read=0;
            start_global_id=drain_stream->audio_circ_buff->byte_id;
            variable_period_size=period_size_in_bytes;
         } 
         if(retval<0){
            LOGMESSAGE("Failure reading circ_buff\n",LOGSTDERR);
            free(drain_audio);
            drain_stream->closing=1;
            return NULL;
         }
      }
      total_bytes_read+=retval;
         

   }

   LOGMESSAGE("Drain worker finished\n",LOGSTDOUT);
   free(drain_audio);     
   return NULL;

}


/** Handles adding a new stream to alsad_stream_list_t.
 * 
 * This function requests an alsad_stream_props from the connected client, 
 * verifying the format, and attempts to add it into the global stream list.
 * If the insertion is successful it will then create a thread to drain the
 * audio buffer (alsad_drain_buffer).
 * 
 * This function allocates an alsad_stream_props_t and an alsad_stream_node_t
 * that get freed by alsad_handle_del_stream.
 * 
 * @param sock the socket descriptor for the current connection
 * @param head_of_list the global list of streams for alsad
 * @param load_keys the key-value pairs loaded from the alsad config file
 * @see alsad_handle_del_stream()
 * @see alsad_drain_buffer()
 * @return An integer -1 on failure or 0 on success
 * 
 */
int alsad_handle_add_stream(int sock, alsad_stream_list_t *head_of_list,
                                               alsad_config_keys_t *load_keys,
                                               alsad_control_t *return_ctrl){


   int retval;
   unsigned int bits_per_sample;
   alsad_net_structs_t recv_structs;
   alsad_stream_props_t rcvd_stream_props; 
   alsad_control_t request_ctrl;
   alsad_drain_buff_args_t *new_drain_args;
   pthread_t thread_ident;
   alsad_stream_node_t *new_stream_node, *working_stream;
     
   
   recv_structs.hw_params=NULL;
   recv_structs.sw_params=NULL;
   recv_structs.local_stream=&rcvd_stream_props;
   recv_structs.remote_stream=NULL;
   recv_structs.connect_addr=NULL;
   recv_structs.local_pipe=NULL;
   recv_structs.remote_pipe=NULL;

   request_ctrl.ctrl=ALSAD_SEND_LOCAL_STREAM;

   if((retval=alsad_request_structs(sock,&recv_structs,&request_ctrl)) < 0){ 
      LOGMESSAGE("Could not receive required structures\n",LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_INSUF_INFO;
      return 0;
   }
   
   
   if((bits_per_sample=snd_pcm_format_physical_width(
                                          rcvd_stream_props.format)) < 0){
      LOGMESSAGE("Physical Format Invalid\n",LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_INVLD_FRMT;
      return 0;
   }

   if(rcvd_stream_props.channels < 1){
      LOGMESSAGE("Must have at least one channel\n",LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_INSUF_INFO;
      return 0;
   }

   rcvd_stream_props.frame_size=(bits_per_sample*
                             rcvd_stream_props.channels)/8;

   fprintf(stdout,"Identifier: %d  Text: %s  Buffer Size: %u\n",rcvd_stream_props.identifier,
                                       rcvd_stream_props.text_desc,
                                       rcvd_stream_props.circ_buff_size);
   
   if((rcvd_stream_props.src_permission & (ALSAD_SRC_CALC_RATE |
                                           ALSAD_SRC_SINK_RATE)) == 0){
      LOGMESSAGE("Could not receive required structures\n",LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_SRC_PERM;
      return 0;
   }   


   if(rcvd_stream_props.rate < 1){
      LOGMESSAGE("Rate must be at least 1.\n",LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_INSUF_INFO;
      return 0;
   }
   
   if((retval=alsad_stream_node_init(&new_stream_node, &rcvd_stream_props,
                                 load_keys)) < 0){
      LOGMESSAGE("Unable to initialize stream", LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_INTERNAL;
      return 0;
   }

   if((working_stream=alsad_stream_list_insert(head_of_list,
                                              new_stream_node))==NULL){
      LOGMESSAGE("Unable to insert stream into list", LOGSTDERR);
      alsad_stream_node_destroy(new_stream_node);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_STREAM_INS;
      return 0;
   }


   new_drain_args=(alsad_drain_buff_args_t *)malloc(
                                             sizeof(alsad_drain_buff_args_t));

   if(new_drain_args==NULL){
      LOGMESSAGE("Could not allocate memory for funct_args",LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_INTERNAL;
      return 0;
   }
   new_drain_args->stream=new_stream_node;
   new_drain_args->list=head_of_list;
   
   retval=pthread_create(&thread_ident,NULL,alsad_drain_buffer,
                               (void *)new_drain_args);
   if(retval!=0){
      LOGMESSAGE("Could not create a thread",LOGSTDERR);
      working_stream=alsad_stream_list_remove(head_of_list,
                            working_stream->stream_props->identifier);
      if(working_stream==NULL){
         LOGMESSAGE("Could not remove list item from list",LOGSTDERR);
      }
      alsad_stream_node_destroy(new_stream_node);
      free(new_drain_args);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_INTERNAL;
      return 0;
   }

   fprintf(stderr,"Thread Created\n");

   return_ctrl->ctrl=ALSAD_MSG_RQST_OK;
   return 0;

}


/** Handles deleting a stream from alsad_stream_list_t.
 * 
 * More detail here
 * 
 * This function frees the alsad_stream_props_t and alsad_stream_node_t allocated
 * by alsad_handle_add_stream.
 * 
 * @param sock the socket descriptor for the current connection
 * @param head_of_list the global list of streams for alsad
 * @see alsad_handle_add_stream()
 * @return An integer -1 on failure or 0 on success
 * 
 */
int alsad_handle_del_stream(int sock, alsad_stream_list_t *head_of_list,
                          alsad_control_t *return_ctrl){


   int retval;
   alsad_net_structs_t recv_structs;
   alsad_control_t request_ctrl;
   alsad_stream_props_t rcvd_stream_props;
   alsad_stream_node_t *ret_stream_ptr;

   recv_structs.hw_params=NULL;
   recv_structs.sw_params=NULL;
   recv_structs.local_stream=&rcvd_stream_props;
   recv_structs.remote_stream=NULL;
   recv_structs.connect_addr=NULL;
   recv_structs.local_pipe=NULL;
   recv_structs.remote_pipe=NULL;

   request_ctrl.ctrl=ALSAD_SEND_LOCAL_STREAM;

   if((retval=alsad_request_structs(sock,&recv_structs,&request_ctrl)) < 0){ 
      LOGMESSAGE("Could not receive required structures\n",LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_INSUF_INFO;
      return 0;
   }

   if((ret_stream_ptr=alsad_stream_list_find(head_of_list,
                                rcvd_stream_props.identifier)) == NULL){
      LOGMESSAGE("Unable to find stream", LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_STREAM_FIND;
      return 0;
   }

   ret_stream_ptr->closing=1;
   
   sleep(ALSAD_THREAD_CHANNEL_WAIT);
   LOGMESSAGE("Woke up and leaving",LOGSTDOUT);
   ret_stream_ptr=alsad_stream_list_remove(head_of_list,
                             rcvd_stream_props.identifier);
   if(ret_stream_ptr==NULL){
      LOGMESSAGE("Could not remove list item from list",LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_STREAM_REM;
      return 0;
   }
   LOGMESSAGE("Removed list item from list",LOGSTDERR);
   alsad_stream_node_destroy(ret_stream_ptr);
   
   return_ctrl->ctrl=ALSAD_MSG_RQST_OK;
   return 0;
}










/** Handles adding a new source to an alsad stream
 * 
 * More detail here
 * 
 * @param sock the socket descriptor for the current connection
 * @param head_of_list the global list of streams for alsad
 * @param rcvd_control the alsad_control_t received from the connected client
 * @param load_keys the key-value pairs loaded from the alsad config file
 * @see alsad_handle_del_src()
 * @return An integer -1 on failure or 0 on success
 * 
 */
int alsad_handle_add_src(int *sock, alsad_stream_list_t *head_of_list,
                                        alsad_control_t *rcvd_control,
                                        alsad_config_keys_t *load_keys,
                                        alsad_control_t *return_ctrl){

   int retval, from_socket, called_thread;
   int rmte_sock;
   alsad_net_structs_t recv_structs, send_structs;
   alsad_hw_params_t rcvd_hw_params; 
   alsad_sw_params_t rcvd_sw_params; 
   alsad_connect_addr_t rcvd_connect_addr;
   alsad_connect_addr_t send_conn_addr;
   alsad_stream_props_t rcvd_stream_props; 
   alsad_stream_props_t rcvd_remote_stream; 
   alsad_data_pipe_t rcvd_local_pipe;
   alsad_data_pipe_t rcvd_remote_pipe;
   alsad_audio_hdr_t rcvd_audio_hdr;
   alsad_control_t request_ctrl,send_control;
   alsad_stream_node_t *working_stream;
   alsad_data_pipe_node_t *new_data_pipe_node;
   alsad_data_pipe_node_t *node_pipe_ptr;
   //unsigned int bits_per_sample;
   snd_pcm_t *handle;
   char audiobuf[ALSAD_MAX_TRANS_SIZE];
   long rcvd_bytes, written_bytes;
   long longretval;


   struct sockaddr_in localaddress;
   int len_sockaddr;
   
   struct timespec timeout;
   
   timeout.tv_sec=ALSAD_DEF_CIRC_TIMEOUT;
   timeout.tv_nsec=0;
   
    
   // Initialize all the pointers and control values 
   len_sockaddr=sizeof(struct sockaddr_in);


   called_thread=0;
   rcvd_bytes=0;
   written_bytes=0;
   request_ctrl.ctrl=0;
   from_socket=0;
   handle=NULL;

   alsad_initialize_hw_params(&rcvd_hw_params); 
   alsad_initialize_sw_params(&rcvd_sw_params); 
   alsad_initialize_connect_addr(&rcvd_connect_addr); 
   alsad_initialize_stream_props(&rcvd_stream_props); 
   alsad_initialize_stream_props(&rcvd_remote_stream); 
   alsad_initialize_data_pipe(&rcvd_remote_pipe); 
   alsad_initialize_data_pipe(&rcvd_local_pipe); 

   
   if((rcvd_control->ctrl | ALSAD_RQST_ADD_SRC)==ALSAD_RQST_ADD_SRC){ 
      LOGMESSAGE("Path of data pipe not specified", LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_NO_PATH;
      return 0; 
   } 
           
   // Ask for the structures that are needed for the connection 
   if((rcvd_control->ctrl & ALSAD_PATH_RMTE_SOCK) == ALSAD_PATH_RMTE_SOCK || 
      (rcvd_control->ctrl & ALSAD_PATH_CUR_SOCK) == ALSAD_PATH_CUR_SOCK){
      from_socket=1;
      request_ctrl.ctrl=(ALSAD_SEND_CONN_ADDR);
      
      if((rcvd_control->ctrl & ALSAD_PATH_RMTE_SOCK) == ALSAD_PATH_RMTE_SOCK){
         request_ctrl.ctrl=(request_ctrl.ctrl | ALSAD_SEND_RMTE_PIPE | ALSAD_SEND_RMTE_STREAM);      
      }
      
   }
   else if((rcvd_control->ctrl & ALSAD_PATH_HW_IFACE) == ALSAD_PATH_HW_IFACE){
      request_ctrl.ctrl=(request_ctrl.ctrl | ALSAD_SEND_HW_PARAMS 
                          | ALSAD_SEND_SW_PARAMS);
   }


   
   request_ctrl.ctrl=(request_ctrl.ctrl | ALSAD_SEND_LOCAL_STREAM); 

   request_ctrl.ctrl=(request_ctrl.ctrl | ALSAD_SEND_LOCAL_PIPE); 

   recv_structs.hw_params=&rcvd_hw_params;
   recv_structs.sw_params=&rcvd_sw_params;
   recv_structs.local_stream=&rcvd_stream_props;
   recv_structs.remote_stream=&rcvd_remote_stream;
   recv_structs.connect_addr=&rcvd_connect_addr;
   recv_structs.local_pipe=&rcvd_local_pipe;
   recv_structs.remote_pipe=&rcvd_remote_pipe;
   
   // Request the required structures from the connecting client.
   if((retval=alsad_request_structs(*sock,&recv_structs,&request_ctrl)) < 0){ 
      LOGMESSAGE("Could not receive required structures\n",LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_INSUF_INFO;
      return 0;
   }
  
//   fprintf(stdout,"Identifier: %d  Text %s\n",rcvd_stream_props->identifier,
//                                     rcvd_stream_props->text_desc);
   
   // Make sure that the stream exists
   if((working_stream=alsad_stream_list_find(head_of_list,rcvd_stream_props.identifier))
                                                                      == NULL){
      LOGMESSAGE("Could not find the requested stream in the list\n",
                                                              LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_STREAM_FIND;
      return 0;
   }

   
   //if((bits_per_sample=snd_pcm_format_physical_width(
   //                                       rcvd_stream_props->format)) < 0){
   //   LOGMESSAGE("Physical Format Invalid\n",LOGSTDERR);
   //   alsad_free_net_structs(&recv_structs);
   //   return -1;
   //}

   
   rcvd_local_pipe.connect_addr=&rcvd_connect_addr;
   rcvd_local_pipe.hw_params=&rcvd_hw_params;
   rcvd_local_pipe.sw_params=&rcvd_sw_params;



   // These can't be changed unless the sinks are alerted.
   // They make assumptions on the size of the stream information.
   // This is especially true in the case of a hw_iface
   
   //working_stream->stream_props->format=rcvd_stream_props->format;
   //working_stream->stream_props->channels=rcvd_stream_props->channels;
   //working_stream->stream_props->rate=rcvd_stream_props->rate;

   
   //working_stream->stream_props->frame_size=(bits_per_sample*
   //                          rcvd_stream_props->channels)/8;


   if((retval=alsad_data_pipe_node_init(&new_data_pipe_node,
                                                   &rcvd_local_pipe)) < 0){
      LOGMESSAGE("Could not init data new pipe\n",LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_INTERNAL;
      return 0;
   }


   if((node_pipe_ptr=alsad_data_pipe_list_insert(&working_stream->sources,
                            new_data_pipe_node)) == NULL){
      LOGMESSAGE("Could not insert data pipe into data_pipe list\n",LOGSTDERR);
      alsad_data_pipe_node_destroy(new_data_pipe_node);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_PIPE_INS;
      return 0;
   }

 
  


   if(from_socket){ 
      //Connect to remote socket if RMTE_SOCK
      if((rcvd_control->ctrl & ALSAD_PATH_RMTE_SOCK) == ALSAD_PATH_RMTE_SOCK){ 
         send_control.ctrl=ALSAD_RQST_ADD_SINK;
         send_control.ctrl=send_control.ctrl | ALSAD_PATH_CUR_SOCK;

         if((retval=getsockname(*sock, (struct sockaddr *)&localaddress,
                                                  &len_sockaddr)) < 0){
            LOGMESSAGE("Could not get localaddress.\n", LOGSTDERR);
            if((node_pipe_ptr=alsad_data_pipe_list_remove(&working_stream->sources,
                            rcvd_local_pipe.identifier)) == NULL){
               LOGMESSAGE("Could not remove data pipe from data_pipe list\n",LOGSTDERR);
            }
 
            alsad_data_pipe_node_destroy(new_data_pipe_node);
            return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
            return_ctrl->code=ALSAD_ERR_LOCAL_ADDR;
            return 0;
         }

         // If the hostname returned is too big, the hostname won't be
         // NULL terminated and will fail on the next connection attempt.
         strncpy(send_conn_addr.hostname, inet_ntoa(localaddress.sin_addr),
                  ALSAD_MAX_HOST_LEN-1);

         send_conn_addr.port=server_port;

         fprintf(stdout," local address is %s:%i\n",send_conn_addr.hostname,
                                                     send_conn_addr.port);



         send_structs.hw_params=NULL;
         send_structs.sw_params=NULL;
         send_structs.local_stream=&rcvd_remote_stream;
         send_structs.remote_stream=NULL;
         send_structs.connect_addr=&send_conn_addr;
         send_structs.local_pipe=&rcvd_remote_pipe;
         send_structs.remote_pipe=NULL;
      
         rmte_sock=alsad_conn_with_host(&rcvd_connect_addr,&send_structs,
                              load_keys->config_username,
                              load_keys->config_priv_key_file,
                              &send_control);
         if(rmte_sock<0){
            LOGMESSAGE("Could not connect with remote host.\n",LOGSTDERR);
            if((node_pipe_ptr=alsad_data_pipe_list_remove(&working_stream->sources,
                            rcvd_local_pipe.identifier)) == NULL){
               LOGMESSAGE("Could not remove data pipe from data_pipe list\n",LOGSTDERR);
            }
            alsad_data_pipe_node_destroy(new_data_pipe_node);
            return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
            return_ctrl->code=ALSAD_ERR_RMTE_HOST;
            return 0;
         }
          
      }
      send_control.ctrl=ALSAD_MSG_RQST_OK;
      if((retval=alsad_send_control(*sock, &send_control)) <= 0){
         LOGMESSAGE("Failed to send control message to client", LOGSTDERR);
         close(rmte_sock);
         close(*sock);
         if((node_pipe_ptr=alsad_data_pipe_list_remove(&working_stream->sources,
                            rcvd_local_pipe.identifier)) == NULL){
            LOGMESSAGE("Could not remove data pipe from data_pipe list\n",LOGSTDERR);
         }
         
         alsad_data_pipe_node_destroy(new_data_pipe_node);
         return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
         return -1;
      }
      if((rcvd_control->ctrl & ALSAD_PATH_RMTE_SOCK) == ALSAD_PATH_RMTE_SOCK){
         close(*sock);
         *sock=rmte_sock;
      }
   }
   else{ 
      /*! \\todo
       * When adding a source that is a hw_iface set the mixer to capture from
       * the device specified by the client.
      **/
      
      rcvd_hw_params.stream=SND_PCM_STREAM_CAPTURE;
      
      if((retval = alsad_configure_snd_card(&handle, 
                                      working_stream->stream_props,
                                                  &rcvd_hw_params)) <= 0){
         LOGMESSAGE("Unable to configure soundcard", LOGSTDERR);


         if((node_pipe_ptr=alsad_data_pipe_list_remove(&working_stream->sources,
                            rcvd_local_pipe.identifier)) == NULL){
            LOGMESSAGE("Could not remove data pipe from data_pipe list\n",LOGSTDERR);
         }
         alsad_data_pipe_node_destroy(new_data_pipe_node);
         return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
         return_ctrl->code=ALSAD_ERR_IFACE_CFG;
         return 0;

      }
      
      
      send_control.ctrl=ALSAD_MSG_RQST_OK;
      if((retval=alsad_send_control(*sock, &send_control)) <= 0){
         LOGMESSAGE("Failed to send control message to client", LOGSTDERR);
         close(*sock);
         if((node_pipe_ptr=alsad_data_pipe_list_remove(&working_stream->sources,
                            rcvd_local_pipe.identifier)) == NULL){
            LOGMESSAGE("Could not remove data pipe from data_pipe list\n",LOGSTDERR);
         }
         alsad_data_pipe_node_destroy(new_data_pipe_node);
         return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
         return -1;
      }

      close(*sock);


      
   } 
   


/** The main processing loop should probably be handled in a separate thread 
 * of execution.  This change would allow a session to continue after a sink or
 * source were created  (this doesn't matter when the path is the current 
 * socket)
 */
   //working_stream->audio_buff->byte_id=0;
  
   while(!(new_data_pipe_node->closing) && !(working_stream->closing)
                                   && !(head_of_list->shutdown)){
      
      if(from_socket){
         send_control.ctrl=ALSAD_SEND_AUDIO_BUF;
         send_control.code=0; 
         if((working_stream->audio_circ_buff->buff_size/2) >
             working_stream->audio_circ_buff->byte_count){
             send_control.code=((working_stream->audio_circ_buff->buff_size/2)-
                            working_stream->audio_circ_buff->byte_count);
             send_control.code=send_control.code - (send_control.code%
                               working_stream->stream_props->frame_size);
             if(send_control.code > ALSAD_MAX_TRANS_SIZE){
                send_control.code=ALSAD_MAX_TRANS_SIZE-
                                  (ALSAD_MAX_TRANS_SIZE % 
                                   working_stream->stream_props->frame_size);
             }
         }                  
         else{
            if((retval=circ_buff_wait_for_read(working_stream->audio_circ_buff,
                                               &timeout)) == 0){
            //LOGMESSAGE("Data removed from circ_buff. Asking for more data from client.", LOGSTDERR);
            }
            continue;
         }

         if((retval=alsad_send_control(*sock, &send_control)) <= 0){
            LOGMESSAGE("Failed to send control message to client", LOGSTDERR);
            new_data_pipe_node->closing=1;
            close(*sock);
            from_socket=1; // This variable is set so that a return value is not
                           // sent to the client and the pipe is remove from the
                           // list
            continue;
         }
         rcvd_bytes=0;
         while(rcvd_bytes < send_control.code && !(new_data_pipe_node->closing)
                                   && !(working_stream->closing)
                                   && !(head_of_list->shutdown)){      

            if((retval=alsad_recv_control(*sock, rcvd_control)) <= 0){
               LOGMESSAGE("Failed to receive control message from client"
                                                              , LOGSTDERR);
               
               new_data_pipe_node->closing=1;
               return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
               return_ctrl->code=ALSAD_ERR_RECV_CTRL;
               break;
            }
            if((rcvd_control->ctrl & ALSAD_RECV_AUDIO_BUF)!=ALSAD_RECV_AUDIO_BUF){
               if((rcvd_control->ctrl & ALSAD_MSG_CLOSE_CONN)!=ALSAD_MSG_CLOSE_CONN){
                  if((rcvd_control->ctrl & ALSAD_MSG_KEEP_ALIVE)==ALSAD_MSG_KEEP_ALIVE){
                     continue;
                  }
                  else{ 
                     LOGMESSAGE("Protocol Error expecting recv audio", LOGSTDERR);
                  }
               }
               else{
                  LOGMESSAGE("Closing connection", LOGSTDOUT);
               }
               new_data_pipe_node->closing=1;
               return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
               return_ctrl->code=ALSAD_ERR_INVLD_CTRL;
               break;
            }

            if((retval=alsad_recv_audio_hdr(*sock, &rcvd_audio_hdr)) < 0){
               LOGMESSAGE("Could not recv_audio_hdr", LOGSTDERR);
               new_data_pipe_node->closing=1;
               return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
               return_ctrl->code=ALSAD_ERR_RECV_AUDIOHD;
               break;
            }

            if(rcvd_audio_hdr.payload_size+rcvd_bytes > send_control.code){
               LOGMESSAGE("Received message greater than amount requested.",
                                                                    LOGSTDERR);
               new_data_pipe_node->closing=1;
               return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
               return_ctrl->code=ALSAD_ERR_MAX_TRANS;
               break;
            }

            if(rcvd_audio_hdr.payload_size % 
                              working_stream->stream_props->frame_size != 0){
               LOGMESSAGE("Received data is not divisable by the frame size.",
                                                                    LOGSTDERR);
               new_data_pipe_node->closing=1;
               return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
               return_ctrl->code=ALSAD_ERR_FRAME_DIV;
               break;
            }
 
            if((retval=safe_sock_recv(*sock, audiobuf,
                                    rcvd_audio_hdr.payload_size, 
                                    ALSAD_PAYLOAD_SOCK_TIMEOUT)) <= 0){
               LOGMESSAGE("safe_sock_recv failed to recv", LOGSTDERR);
               new_data_pipe_node->closing=1;
               return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
               return_ctrl->code=ALSAD_ERR_RECV_AUDIO;
               break;
            }
            //Even if I set up mutex on the circ_buff I still need to
            //check to make sure that the channel and pipe are not closing
            retval=0;
            written_bytes=0;
            while(rcvd_audio_hdr.payload_size > written_bytes && 
                           !working_stream->closing &&
                           !new_data_pipe_node->closing &&
                                            !head_of_list->shutdown){

               longretval=circ_buff_write(working_stream->audio_circ_buff,
                                      audiobuf+written_bytes,
                                      (long)(rcvd_audio_hdr.payload_size-
                                      written_bytes),
                                      &timeout);
               if(longretval<0){
                  LOGMESSAGE("Could not write to circ_buff", LOGSTDERR);
                  new_data_pipe_node->closing=1;
                  return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
                  return_ctrl->code=ALSAD_ERR_INTERNAL;
                  break;
               }
               written_bytes+=longretval;
               rcvd_bytes+=longretval;
            }
         } 
      }
      else{
         //from_socket==0
      
         if((retval = alsad_hw_capture(handle,working_stream->stream_props,
                                         &rcvd_hw_params,audiobuf)) < 0){
            LOGMESSAGE("Failed to capture from card", LOGSTDERR);
            if(handle!=NULL)
               snd_pcm_close(handle);
            if((retval = alsad_configure_snd_card(&handle, 
                                      working_stream->stream_props,
                                                  &rcvd_hw_params)) <= 0){
               LOGMESSAGE("Unable to configure soundcard", LOGSTDERR);
               return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
               return_ctrl->code=ALSAD_ERR_IFACE_CFG;
               break;
            }
            continue;
         }
         retval=0;
         rcvd_bytes=0;
         while(working_stream->stream_props->frame_size*
               rcvd_hw_params.period_size > rcvd_bytes &&
                           !working_stream->closing &&
                           !new_data_pipe_node->closing &&
                                            !head_of_list->shutdown){

            retval=circ_buff_write(working_stream->audio_circ_buff,
                      audiobuf+rcvd_bytes,
                      (rcvd_hw_params.period_size*
                      working_stream->stream_props->frame_size)-
                      rcvd_bytes,
                      &timeout);
            if(retval<0){
               LOGMESSAGE("Could not write to circ_buff", LOGSTDERR);
               new_data_pipe_node->closing=1;
               return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
               return_ctrl->code=ALSAD_ERR_INTERNAL;
               break;
            }
            rcvd_bytes+=retval;
         }
      }

      if(new_data_pipe_node->closing){
         LOGMESSAGE("Pipe closed by flag", LOGSTDOUT);
         break;
      }

      if(working_stream->closing){
         LOGMESSAGE("Stream Closed Remotely", LOGSTDOUT);
         return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
         return_ctrl->code=ALSAD_ERR_STREAM_CLOSE;
         break;
      }

      if(head_of_list->shutdown){
         LOGMESSAGE("alsad shutdown detected... exiting", LOGSTDOUT);
         return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
         return_ctrl->code=ALSAD_ERR_SHUTDOWN;
         break;
      }
     
   }
   LOGMESSAGE("Broke out of loop", LOGSTDERR);

   new_data_pipe_node->closing=1;
   
//   sleep(ALSAD_THREAD_streamNEL_WAIT);
 
   if(handle!=NULL)
      snd_pcm_close(handle);
      

   if((node_pipe_ptr=alsad_data_pipe_list_remove(&working_stream->sources,
                            rcvd_local_pipe.identifier)) == NULL){
      LOGMESSAGE("Could not remove data pipe from data_pipe list\n",LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_PIPE_REM;
   }
   alsad_data_pipe_node_destroy(new_data_pipe_node);
   
   if(from_socket)
      return 0;
   else
      return -1;
      
}




/** Handles deleting a source from an alsad stream
 * 
 * More description here
 * 
 * @param sock the socket descriptor for the current connection
 * @param head_of_list the global list of streams for alsad
 * @param rcvd_control the alsad_control_t received from the connected client
 * @see alsad_handle_add_src()
 * @return An integer -1 on failure or 0 on success
 * 
 */
int alsad_handle_del_src(int sock, alsad_stream_list_t *head_of_list,
                                             alsad_control_t *rcvd_control,
                                             alsad_control_t *return_ctrl){

   int retval;
   
   alsad_data_pipe_t rcvd_local_pipe;
   alsad_stream_props_t rcvd_stream_props;
   alsad_control_t request_ctrl;
   alsad_net_structs_t recv_structs;
   alsad_stream_node_t *working_stream;
   alsad_data_pipe_node_t *working_data_pipe;
   
   request_ctrl.ctrl=(ALSAD_SEND_LOCAL_PIPE | ALSAD_SEND_LOCAL_STREAM); 
   
   recv_structs.hw_params=NULL;
   recv_structs.sw_params=NULL;
   recv_structs.local_stream=&rcvd_stream_props;
   recv_structs.remote_stream=NULL;
   recv_structs.connect_addr=NULL;
   recv_structs.local_pipe=&rcvd_local_pipe;
   recv_structs.remote_pipe=NULL;
  
   // Request the required structures from the connecting client.
   if((retval=alsad_request_structs(sock,&recv_structs,&request_ctrl)) < 0){ 
      LOGMESSAGE("Could not receive required structures\n",LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_INSUF_INFO;
      return 0;
   }
   
   // Make sure that the stream exists
   if((working_stream=alsad_stream_list_find(head_of_list,
                                          rcvd_stream_props.identifier))
                                                                      == NULL){
      LOGMESSAGE("Could not find the requested stream in the list\n",
                                                              LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_STREAM_FIND;
      return 0;
   }
 
   if((working_data_pipe=alsad_data_pipe_list_find(&working_stream->sources,
                            rcvd_local_pipe.identifier)) == NULL){
      LOGMESSAGE("Could not find data pipe in linked list\n",LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_PIPE_FIND;
      return 0;
   }
  
   working_data_pipe->closing=1;

   return_ctrl->ctrl=ALSAD_MSG_RQST_OK;
   return 0;

}







/** Handles adding a sink to an alsad stream
 * 
 * More description here
 * 
 * @param sock the socket descriptor for the current connection
 * @param head_of_list the global list of streams for alsad
 * @param rcvd_control the alsad_control_t received from the connected client
 * @param load_keys the key-value pairs loaded from the alsad config file
 * @see alsad_handle_del_sink()
 * @return An integer -1 on failure or 0 on success
 * 
 */
int alsad_handle_add_sink(int *sock, alsad_stream_list_t *head_of_list,
                                    alsad_control_t *rcvd_control, 
                                    alsad_config_keys_t *load_keys,
                                    alsad_control_t *return_ctrl){
        
   int retval, read_error;
   int to_socket, rmte_sock;
   long sent_bytes, read_bytes;
   alsad_net_structs_t recv_structs, send_structs;
   alsad_hw_params_t rcvd_hw_params; 
   alsad_sw_params_t rcvd_sw_params; 
   alsad_connect_addr_t rcvd_connect_addr; 
   alsad_stream_props_t rcvd_stream_props;
   alsad_stream_props_t rcvd_remote_stream;
   alsad_data_pipe_t rcvd_local_pipe;
   alsad_data_pipe_t rcvd_remote_pipe;
   alsad_data_pipe_node_t *node_pipe_ptr;
   alsad_data_pipe_node_t *new_data_pipe_node;
   alsad_audio_hdr_t send_audio_hdr;
   alsad_control_t request_ctrl,send_control;
   alsad_stream_node_t *working_stream;
   snd_pcm_t *handle;
   char audiobuf[ALSAD_MAX_TRANS_SIZE];
   long thread_read_ptr, longretval;
   
   struct timespec timeout;
   
      
   timeout.tv_sec=ALSAD_DEF_CIRC_TIMEOUT;
   timeout.tv_nsec=0;
   
   // Initialize all the pointers and control values 
   request_ctrl.ctrl=0;
   to_socket=0;
   sent_bytes=0;
   read_bytes=0;
   handle=NULL;
   thread_read_ptr=0;
   longretval=0;

   alsad_initialize_hw_params(&rcvd_hw_params); 
   alsad_initialize_sw_params(&rcvd_sw_params); 
   alsad_initialize_connect_addr(&rcvd_connect_addr); 
   alsad_initialize_stream_props(&rcvd_stream_props); 
   alsad_initialize_stream_props(&rcvd_remote_stream); 
   alsad_initialize_data_pipe(&rcvd_remote_pipe); 
   alsad_initialize_data_pipe(&rcvd_local_pipe); 


   if((rcvd_control->ctrl | ALSAD_RQST_ADD_SINK)==ALSAD_RQST_ADD_SINK){ 
      LOGMESSAGE("Path of data pipe not specified", LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_NO_PATH;
      return 0; 
   } 
           

   if((rcvd_control->ctrl & ALSAD_PATH_RMTE_SOCK) == ALSAD_PATH_RMTE_SOCK || 
      (rcvd_control->ctrl & ALSAD_PATH_CUR_SOCK) == ALSAD_PATH_CUR_SOCK){
      to_socket=1;
      request_ctrl.ctrl=ALSAD_SEND_CONN_ADDR;
   }
   
   if((rcvd_control->ctrl & ALSAD_PATH_RMTE_SOCK) == ALSAD_PATH_RMTE_SOCK){
      request_ctrl.ctrl=(request_ctrl.ctrl | ALSAD_SEND_RMTE_PIPE |
                                                 ALSAD_SEND_RMTE_STREAM);
   }


   if((rcvd_control->ctrl & ALSAD_PATH_HW_IFACE) == ALSAD_PATH_HW_IFACE){
      request_ctrl.ctrl=(request_ctrl.ctrl | ALSAD_SEND_HW_PARAMS 
                          | ALSAD_SEND_SW_PARAMS);
   }

   request_ctrl.ctrl=(request_ctrl.ctrl | ALSAD_SEND_LOCAL_STREAM); 
   request_ctrl.ctrl=(request_ctrl.ctrl | ALSAD_SEND_LOCAL_PIPE); 
   
   
   recv_structs.hw_params=&rcvd_hw_params;
   recv_structs.sw_params=&rcvd_sw_params;
   recv_structs.local_stream=&rcvd_stream_props;
   recv_structs.remote_stream=&rcvd_remote_stream;
   recv_structs.connect_addr=&rcvd_connect_addr;
   recv_structs.local_pipe=&rcvd_local_pipe;
   recv_structs.remote_pipe=&rcvd_remote_pipe;
   
   // Request the required structures from the connecting client.
   if((retval=alsad_request_structs(*sock,&recv_structs,&request_ctrl)) < 0){ 
      LOGMESSAGE("Could not receive required structures\n",LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_INSUF_INFO;
      return 0;
   }
  
   // Make sure that the stream exists
   if((working_stream=alsad_stream_list_find(head_of_list,
                            recv_structs.local_stream->identifier))
                                                                      == NULL){
      LOGMESSAGE("Could not find the requested stream in the list\n",
                                                              LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_STREAM_FIND;
      return 0;
   }

   
   rcvd_local_pipe.connect_addr=&rcvd_connect_addr;
   rcvd_local_pipe.hw_params=&rcvd_hw_params;
   rcvd_local_pipe.sw_params=&rcvd_sw_params;
   

   if((retval=alsad_data_pipe_node_init(&new_data_pipe_node,
                                                   &rcvd_local_pipe)) < 0){
      LOGMESSAGE("Could not init data new pipe\n",LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_INTERNAL;
      return 0;
   }


   if((node_pipe_ptr=alsad_data_pipe_list_insert(&working_stream->sinks,
                            new_data_pipe_node)) == NULL){
      LOGMESSAGE("Could not insert data pipe into data_pipe list\n",LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_PIPE_INS;
      return 0;
   }
   

   if(to_socket){ 
      
      //Connect to remote socket if RMTE_SOCK
      if((rcvd_control->ctrl & ALSAD_PATH_RMTE_SOCK) == ALSAD_PATH_RMTE_SOCK){
         LOGMESSAGE("Adding a sink from a remote socket\n",LOGSTDERR); 
         send_control.ctrl=ALSAD_RQST_ADD_SRC;
         send_control.ctrl=send_control.ctrl | ALSAD_PATH_CUR_SOCK;

         send_structs.hw_params=NULL;
         send_structs.sw_params=NULL;
         send_structs.local_stream=&rcvd_remote_stream;
         send_structs.remote_stream=NULL;
         send_structs.connect_addr=&rcvd_connect_addr;
         send_structs.local_pipe=&rcvd_remote_pipe;
         send_structs.remote_pipe=NULL;
      
         rmte_sock=alsad_conn_with_host(&rcvd_connect_addr,&send_structs,
                              load_keys->config_username,
                              load_keys->config_priv_key_file,
                              &send_control);
         if(rmte_sock==0){
             LOGMESSAGE("Could not connect with remote host.\n",LOGSTDERR);
             if((node_pipe_ptr=alsad_data_pipe_list_remove(&working_stream->sinks,
                            rcvd_local_pipe.identifier)) == NULL){
               LOGMESSAGE("Could not remove data pipe from data_pipe list\n",LOGSTDERR);
      
             }
             return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
             return_ctrl->code=ALSAD_ERR_RMTE_HOST;
             return 0;
             
         }
          
      }
      send_control.ctrl=ALSAD_MSG_RQST_OK;
      if((retval=alsad_send_control(*sock, &send_control)) <= 0){
         LOGMESSAGE("Failed to send control message to client", LOGSTDERR);
         close(rmte_sock);
         close(*sock);
         if((node_pipe_ptr=alsad_data_pipe_list_remove(&working_stream->sinks,
                            rcvd_local_pipe.identifier)) == NULL){
            LOGMESSAGE("Could not remove data pipe from data_pipe list\n",LOGSTDERR);
   
         }
         return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
         return -1;
         
      }
      if((rcvd_control->ctrl & ALSAD_PATH_RMTE_SOCK) == ALSAD_PATH_RMTE_SOCK){
         close(*sock);
         *sock=rmte_sock;
      }
   }
   else{ 
      
      


      LOGMESSAGE("Adding a Hardware Sink\n",LOGSTDERR);
      rcvd_hw_params.stream=SND_PCM_STREAM_PLAYBACK;
      if((retval = alsad_configure_snd_card(&handle, 
                                      working_stream->stream_props,
                                                  &rcvd_hw_params)) <= 0){
         LOGMESSAGE("Unable to configure soundcard", LOGSTDERR);
         if((node_pipe_ptr=alsad_data_pipe_list_remove(&working_stream->sinks,
                            rcvd_local_pipe.identifier)) == NULL){
            LOGMESSAGE("Could not remove data pipe from data_pipe list\n",LOGSTDERR);
   
         }
         return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
         return_ctrl->code=ALSAD_ERR_IFACE_CFG;
         return 0;
      }
 
      if(new_data_pipe_node->closing!=1){

         send_control.ctrl=ALSAD_MSG_RQST_OK;
         if((retval=alsad_send_control(*sock, &send_control)) <= 0){
            LOGMESSAGE("Failed to send control message to client", LOGSTDERR);
            close(*sock);
            if((node_pipe_ptr=alsad_data_pipe_list_remove(&working_stream->sinks,
                            rcvd_local_pipe.identifier)) == NULL){
               LOGMESSAGE("Could not remove data pipe from data_pipe list\n",LOGSTDERR);
   
            }
            return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
            return -1;
         }
         close(*sock);
      }
   } 
   

   while(!(new_data_pipe_node->closing) && !(working_stream->closing) &&
                                      !(head_of_list->shutdown)){
      if((retval=circ_buff_init_thread(working_stream->audio_circ_buff,&thread_read_ptr,
                                   &new_data_pipe_node->thread_byte_id,
                                   &timeout)) < 0){
         if(to_socket){ 
            send_control.ctrl=ALSAD_MSG_KEEP_ALIVE;
            //LOGMESSAGE("Sending Keep Alive Message", LOGSTDERR);
            if((retval = alsad_send_control(*sock,&send_control)) <= 0){
               LOGMESSAGE("Failed to send keep alive to client", LOGSTDERR);
               new_data_pipe_node->closing=1; 
               to_socket=0;  // So I will return -1
               break;
            } 
         }
         continue; 
      }
      break;
   } 



   while(!(new_data_pipe_node->closing) && !(working_stream->closing) &&
                                      !(head_of_list->shutdown)){
      
      if(to_socket){    // Sink is going to another socket
         //LOGMESSAGE("Waiting for control message", LOGSTDERR);
         if((retval=alsad_recv_control(*sock, rcvd_control)) <= 0){
            LOGMESSAGE("Failed to receive control message from client", LOGSTDERR);
            return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
            return_ctrl->code=ALSAD_ERR_RECV_CTRL;
            break;
         }
          
         //LOGMESSAGE("Received control message", LOGSTDERR);
         if((rcvd_control->ctrl & ALSAD_SEND_AUDIO_BUF) != ALSAD_SEND_AUDIO_BUF){
            if((rcvd_control->ctrl & ALSAD_MSG_CLOSE_CONN) == ALSAD_MSG_CLOSE_CONN){
               LOGMESSAGE("Client requested that the connection close", LOGSTDERR);
               return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
               return_ctrl->code=ALSAD_ERR_PIPE_CLOSE;
               break;
            }
            LOGMESSAGE("Unexpected request closing connection", LOGSTDERR);
            return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
            return_ctrl->code=ALSAD_ERR_INVLD_CTRL;
            break;
         }

         if(rcvd_control->code > ALSAD_MAX_TRANS_SIZE){
            LOGMESSAGE("Requested transmission size larger than MAX", LOGSTDERR);
            return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
            return_ctrl->code=ALSAD_ERR_MAX_TRANS;
            break;
         }

         sent_bytes=0;
         while(sent_bytes < rcvd_control->code &&
                   !(new_data_pipe_node->closing) &&
                   !(working_stream->closing) &&
                   !(head_of_list->shutdown)){
            longretval=0;
            //fprintf(stderr, "thread_read_ptr %ld      byte_id %llu\n",thread_read_ptr, new_data_pipe_node->thread_byte_id);
            if((longretval=circ_buff_read_thread(working_stream->audio_circ_buff,
                                       audiobuf+sent_bytes,
                                      &thread_read_ptr,
                                      &new_data_pipe_node->thread_byte_id,
                                      rcvd_control->code-sent_bytes,
                                      &timeout))<=0){
               LOGMESSAGE("Error reading from thread circ buff", LOGSTDERR);
               while(!(new_data_pipe_node->closing) &&
                     !(working_stream->closing) &&
                                      !(head_of_list->shutdown)){
                  if((retval=circ_buff_init_thread(working_stream->audio_circ_buff,
                                   &thread_read_ptr,
                                   &new_data_pipe_node->thread_byte_id,
                                   &timeout)) < 0){
                     send_control.ctrl=ALSAD_MSG_KEEP_ALIVE;
                     //LOGMESSAGE("Sending Keep Alive Message", LOGSTDERR);
                     if((retval = alsad_send_control(*sock,&send_control)) <= 0){
                        LOGMESSAGE("Failed to send keep alive to client", LOGSTDERR);
                        new_data_pipe_node->closing=1; 
                        to_socket=0;  // So I will return -1
                        break;
                     } 
                     continue; 
                  }
                  break;
               }
            }
            if(longretval<=0){
               send_control.ctrl=ALSAD_MSG_KEEP_ALIVE;
               //LOGMESSAGE("Sending Keep Alive Message", LOGSTDERR);
               if((retval = alsad_send_control(*sock,&send_control)) <= 0){
                  LOGMESSAGE("Failed to send keep alive to client", LOGSTDERR);
                  new_data_pipe_node->closing=1; 
                  to_socket=0;  // So I will return -1
                  break;
               } 
               continue;
            }
            send_audio_hdr.payload_size = longretval;
            if((retval = alsad_send_audio_hdr(*sock, &send_audio_hdr,
                               audiobuf+sent_bytes)) == send_audio_hdr.payload_size){
               LOGMESSAGE("Failed to send data to client", LOGSTDERR);
               new_data_pipe_node->closing=1; 
               to_socket=0;  // So I will return -1
               break;
            } 
            sent_bytes+=longretval;
            
            //fprintf(stderr,"Payload size: %ld\n",longretval);
         }
      }
      else{   // Sink is a hw_iface
         
         
         read_bytes=0;
         if(rcvd_hw_params.period_size* working_stream->stream_props->frame_size <= read_bytes){
                 fprintf(stderr,"State Failure......\n");
                 fprintf(stderr,"period_size: %lu\n",rcvd_hw_params.period_size);
                 fprintf(stderr,"frame_size: %u\n",working_stream->stream_props->frame_size );
         }
         while(read_bytes < rcvd_hw_params.period_size*
                                   working_stream->stream_props->frame_size &&
                   !(new_data_pipe_node->closing) &&
                   !(working_stream->closing) &&
                   !(head_of_list->shutdown)){
            longretval=0;

            if((longretval=circ_buff_read_thread(
                               working_stream->audio_circ_buff,
                               audiobuf+read_bytes, &thread_read_ptr,
                               &new_data_pipe_node->thread_byte_id,
                               (rcvd_hw_params.period_size*
                               working_stream->stream_props->frame_size)-
                               read_bytes, &timeout))<=0){
               LOGMESSAGE("Error reading from thread circ buff", LOGSTDERR);
               while(!(new_data_pipe_node->closing) &&
                     !(working_stream->closing) &&
                                      !(head_of_list->shutdown)){
                  if((retval=circ_buff_init_thread(working_stream->audio_circ_buff,
                                   &thread_read_ptr,
                                   &new_data_pipe_node->thread_byte_id,
                                   &timeout)) < 0){
                     continue; 
                  }
                  break;
               }
            }

            if(longretval>0){
               read_bytes+=longretval;
            //fprintf(stderr, "thread_read_ptr %lu\tbyte_id %llu\tglobal_byte_id %llu\tbyte_count %ld\tread_bytes %ld\n",
            //                         thread_read_ptr, new_data_pipe_node->thread_byte_id,
            //                         working_stream->audio_circ_buff->byte_id,
            //                         working_stream->audio_circ_buff->byte_count,
            //                         read_bytes);

            }
         }

         
         if((retval = alsad_hw_playback(handle,working_stream->stream_props,
                                          &rcvd_hw_params,audiobuf)) < 0 &&
                                          !read_error){
            LOGMESSAGE("Failed to playback sound data", LOGSTDERR);
            if(handle!=NULL)
               snd_pcm_close(handle);
            if((retval = alsad_configure_snd_card(&handle, 
                                      working_stream->stream_props,
                                                  &rcvd_hw_params)) <= 0){
               LOGMESSAGE("Unable to configure soundcard", LOGSTDERR);
               return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
               return_ctrl->code=ALSAD_ERR_IFACE_CFG;
               break;
            }
         }

      }
   }

   LOGMESSAGE("Broke out of loop", LOGSTDERR);

   if((node_pipe_ptr=alsad_data_pipe_list_remove(&working_stream->sinks,
                            rcvd_local_pipe.identifier)) == NULL){
      LOGMESSAGE("Could not remove data pipe from data_pipe list\n",LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_PIPE_REM;
      return 0;
   }
         
   
   if(handle!=NULL)
      snd_pcm_close(handle);
      
   // The only time this should be open is when sending to another socket
   if(to_socket)
      return 0;
   else
      return -1;
   
}






/** Handles deleting a sink to from alsad stream
 * 
 * More description here
 * 
 * @param sock the socket descriptor for the current connection
 * @param head_of_list the global list of streams for alsad
 * @param rcvd_control the alsad_control_t received from the connected client
 * @see alsad_handle_del_sink()
 * @return An integer -1 on failure or 0 on success
 * 
 */
int alsad_handle_del_sink(int sock, alsad_stream_list_t *head_of_list,
                                    alsad_control_t *rcvd_control,
                                    alsad_control_t *return_ctrl){

   int retval;
   alsad_data_pipe_t rcvd_local_pipe;
   alsad_stream_props_t rcvd_stream_props;
   alsad_control_t request_ctrl;
   alsad_net_structs_t recv_structs;
   alsad_stream_node_t *working_stream;
   alsad_data_pipe_node_t *working_data_pipe;

   request_ctrl.ctrl=(ALSAD_SEND_LOCAL_PIPE | ALSAD_SEND_LOCAL_STREAM); 
   
   recv_structs.hw_params=NULL;
   recv_structs.sw_params=NULL;
   recv_structs.local_stream=&rcvd_stream_props;
   recv_structs.remote_stream=NULL;
   recv_structs.connect_addr=NULL;
   recv_structs.local_pipe=&rcvd_local_pipe;
   recv_structs.remote_pipe=NULL;
   
   // Request the required structures from the connecting client.
   if((retval=alsad_request_structs(sock,&recv_structs,&request_ctrl)) < 0){ 
      LOGMESSAGE("Could not receive required structures\n",LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_INSUF_INFO;
      return 0;
   }
   
   
   /**\todo I need a function to perform these operations in alsad_stream_list_t
    */
   
   // Make sure that the stream exists
   if((working_stream=alsad_stream_list_find(
                                head_of_list,rcvd_stream_props.identifier))
                                                                      == NULL){
      LOGMESSAGE("Could not find the requested stream in the list\n",
                                                              LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_STREAM_FIND;
      return 0;
   }
   // Check to see if the data pipe exists so it can be closed.
   
   if((working_data_pipe=alsad_data_pipe_list_find(&working_stream->sinks,
                            rcvd_local_pipe.identifier)) == NULL){
      LOGMESSAGE("Could not find data pipe in linked list\n",LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_PIPE_FIND;
      return 0;
   }
  
   working_data_pipe->closing=1;
  

   LOGMESSAGE("Data pipe set to be closed\n",LOGSTDERR);
  
   return_ctrl->ctrl=ALSAD_MSG_RQST_OK;
   return 0;
   

}




/** Handles returning a list of alsad streams to a connected client
 * 
 * More description here
 * 
 * @param sock the socket descriptor for the current connection
 * @param head_of_list the global list of streams for alsad
 * @return An integer -1 on failure or 0 on success
 * 
 */

int alsad_handle_list_stream(int sock, alsad_stream_list_t *head_of_list,
                                      alsad_control_t *return_ctrl){

   int retval;
   alsad_control_t send_control;

   //Send an OK to get client out of request_net_structs
   send_control.ctrl=ALSAD_MSG_RQST_OK;
   if((retval=alsad_send_control(sock, &send_control)) <= 0){
      LOGMESSAGE("Failed to send control message to client", LOGSTDERR);
      close(sock);
      return -1;
   }
   
   send_control.ctrl=ALSAD_MSG_RQST_OK;
   if((retval=alsad_send_control(sock, &send_control)) <= 0){
      LOGMESSAGE("Failed to send control message to client", LOGSTDERR);
      close(sock);
      return -1;
   }

   if((retval=alsad_stream_list_send(sock, head_of_list)) < 0){
      LOGMESSAGE("Failed to send stream list", LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_INTERNAL;
      return 0;
   } 
   
   return_ctrl->ctrl=ALSAD_MSG_RQST_OK;
   return 0;
}



/** Handles returning a list of alsad_data_pipe_t to a connected client
 * 
 * More description here
 * 
 * @param sock the socket descriptor for the current connection
 * @param head_of_list the global list of streams for alsad
 * @param rcvd_control the alsad_control_t received from the connected client
 * @return An integer -1 on failure or 0 on success
 * 
 */
int alsad_handle_list_pipe(int sock, alsad_stream_list_t *list,
                                    alsad_control_t *rcvd_control,
                                    alsad_control_t *return_ctrl){

   int retval;

   alsad_stream_props_t rcvd_stream_props; 
   alsad_control_t request_ctrl;
   alsad_net_structs_t recv_structs;

   request_ctrl.ctrl=(ALSAD_SEND_LOCAL_STREAM); 
   
   recv_structs.hw_params=NULL;
   recv_structs.sw_params=NULL;
   recv_structs.local_stream=&rcvd_stream_props;
   recv_structs.remote_stream=NULL;
   recv_structs.connect_addr=NULL;
   recv_structs.local_pipe=NULL;
   recv_structs.remote_pipe=NULL;
   
   // Request the required structures from the connecting client.
   if((retval=alsad_request_structs(sock,&recv_structs,&request_ctrl)) < 0){ 
      LOGMESSAGE("Could not receive required structures\n",LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_INSUF_INFO;
      return 0;
   }
  
   if((retval=alsad_stream_list_send_pipes(sock, list, rcvd_stream_props.identifier)) < 0){
      LOGMESSAGE("Could not find the requested stream in the list\n",
                                                              LOGSTDERR);
      return_ctrl->ctrl=ALSAD_MSG_RQST_ERR;
      return_ctrl->code=ALSAD_ERR_INTERNAL;
      return 0;
   }
    
   return_ctrl->ctrl=ALSAD_MSG_RQST_OK;
   return 0;

}


int alsad_is_server(alsad_connect_addr_t *connection){
   if(connection->port!=0)
      return 0;

   return -1;

}


static void catch_signal(int signal){
   LOGMESSAGE("Exiting Carefully..... ", LOGSTDOUT);
   alsad_shutdown=1;

}

/** @}*/
