 /*! \file config_file.c
 *  \brief This is the implementation of the Configuration-File Library
 *  \author Daniel R. Warren
 *  \version 1.0
 *  \date    November 2004
 *  \ingroup config_file_lib
*/


 /** \defgroup config_file_lib The Configuration-File Library
 *   \ingroup helper_components
 *   \brief Library that provides a method for accessing and writing
 *          config files.
 * 
 * It is often important for a program to input data at runtime. 
   This task is often achieved by using a configuration file. 
   The functions provided in this library allow a programmer to pull
   key-value pairs from a configuration file and putting them into memory.
   Once in memory, the list of key-value pairs can be searched by key to
   locate a value. If the list is modified, it can be written back to a
   configuration file. Most configuration files use the format 
   \<key\>=\<value\>.
   
   This library provides the same basic format but is less flexible
   than other applications in that extra spaces and new line characters will
   cause the key-value pairs to be loaded improperly. A programmer should
   strictly adhere to the following format when creating key value pair files:
   
   \verbatim
      <beginning of file><key_1>=<value_1><new line character>
      <key_2>=<value_2><new line character>
      .
      .
      .
      <key_n>=<value_n><end of file>
   \endverbatim
      
      
   Any string of characters (except strings than contain an equals sign)
   with a length greater or equal to 1 can be used as a key or value.
   
   Here is an example:
   
   \include alsad.conf
   
 */
 
/** \addtogroup config_file_lib */
/** @{*/

#include "config_file.h"


key_value_pair_t *config_file_read(char *filename, int *loaded_pairs){


   FILE *file;
   char *buffer;
   key_value_pair_t *new_pairs;
   char* charptr;
   size_t retval, retsize;
   int resize_factor=1;
   *loaded_pairs=0;

   file = fopen(filename, "r");
   if(file == NULL){
      fprintf(stderr, "config_file_read: Could not open config file: %s\n",
		                                                 filename);
      return NULL;
   }

   new_pairs = (key_value_pair_t *) malloc(CONFIG_FILE_MAX_PAIRS*
		                                  sizeof(key_value_pair_t));
   if(new_pairs == NULL){
      fprintf(stderr, "config_file_read: could not malloc new_pairs.\n");
      fclose(file);
      return NULL;
   }
   buffer=NULL;
   while((retval=getline(&buffer,&retsize,file)) != -1 &&
                     resize_factor <= CONFIG_FILE_MAX_RESIZE){
  
      if(*(buffer+(strlen(buffer)-1)) == '\n'){
         *(buffer+(strlen(buffer)-1)) = '\0';
      } 
      charptr = strchr(buffer, '=');
      if(charptr == NULL){
         free(buffer);
	     buffer=NULL;
         break;
      }
      *charptr = '\0';
      charptr++;

      if((*loaded_pairs) >= CONFIG_FILE_MAX_PAIRS*resize_factor){
         resize_factor++;
	     new_pairs=realloc(new_pairs,CONFIG_FILE_MAX_PAIRS*
			            resize_factor*sizeof(key_value_pair_t));
         if(new_pairs == NULL){
            fprintf(stderr, "config_file_read: could not realloc new_pairs\n");
            fclose(file);
            free(buffer);
            return NULL;
         }
      }

      //fprintf(stdout,"key: %s  value: %s\n",buffer,charptr);
      new_pairs[(*loaded_pairs)].key=strdup(buffer);
      new_pairs[(*loaded_pairs)].value=strdup(charptr);
      (*loaded_pairs)++;

      free(buffer);
      buffer=NULL;
   }

   fclose(file);

   return new_pairs;
}


int config_file_write(char *filename, key_value_pair_t *pairs,
                                                 int pairs_to_write){

   FILE* file;
   int   i;

   file = fopen(filename, "w");
   if(file == NULL){
      fprintf(stderr,"config_file_write: Could not open config file:");
      fprintf(stderr," %s for writing.\n", filename);
      return -1;
   }

   for(i=0; i<pairs_to_write; i++){
      fputs(pairs[i].key, file);
      fputs("=", file);
      fputs(pairs[i].value, file);
      fputs("\n", file);
   }

   fflush(file);
   fclose(file);

   return 0;
}


char *config_file_get_value(key_value_pair_t *pairs,char *key, int pairs_loaded){

   int i;

   for(i = 0; i < pairs_loaded; i++){
      if(strcmp(pairs[i].key, key) == 0)
      {
         return pairs[i].value;
      }
   }
   return NULL;

}


void config_file_destroy(key_value_pair_t *pairs, int pairs_loaded){

   int i;

   for(i = 0; i < pairs_loaded; i++){
      free(pairs[i].key);
      free(pairs[i].value);
   }
   free(pairs);

}

/** @}*/
