/*! \file circ_buff.c
 *  \brief This is the implementation of the Circular-Buffer Library
 *  \author Daniel R. Warren
 *  \version 1.0
 *  \date    November 2004
 *  \ingroup circ_buff_lib
*/


/** \defgroup circ_buff_lib The Circular-Buff Library
 *   \ingroup helper_components
 *   \brief Library that provides a method for writing and reading to/from
 *          a circular buffer.
 * 
 * 
 */
 
 /** \addtogroup circ_buff_lib */
/** @{*/
 
 
#include "circ_buff.h"

int circ_buff_init(circ_buff_t **new_buff, long buff_size){

      
   *new_buff=(circ_buff_t *)malloc(sizeof(circ_buff_t));
   if(*new_buff==NULL){   
      fprintf(stderr,"init_circ_buff: Could not ");
      fprintf(stderr,"allocate memory for new_buff\n");  
      return -1;
   }

   (*new_buff)->buff_ptr=(void *)malloc(buff_size);
   if((*new_buff)->buff_ptr == NULL){
      fprintf(stderr,"init_circ_buff: Could not ");
      fprintf(stderr,"allocate memory for buffer\n");
      free(*new_buff);
      return -1;
   }
      
   (*new_buff)->write_pnt=0;
   (*new_buff)->read_pnt=0;
   (*new_buff)->byte_count=0;
   (*new_buff)->byte_id=0;
   (*new_buff)->buff_size=buff_size;
   pthread_mutex_init(&((*new_buff)->circ_write_mut),NULL);
   pthread_mutex_init(&((*new_buff)->circ_read_mut),NULL);
   pthread_mutex_init(&((*new_buff)->circ_size_var_mut),NULL);
 
   pthread_mutex_init(&((*new_buff)->circ_write_cond_mut),NULL);
   pthread_cond_init(&((*new_buff)->circ_write_cond_var),NULL);
 
   pthread_mutex_init(&((*new_buff)->circ_read_cond_mut),NULL);
   pthread_cond_init(&((*new_buff)->circ_read_cond_var),NULL);
         
   return 0;
}





/* In other implementations of the pthread library mutexes and
 * condition variables must be destroyed.  In the LinuxThreads
 * implementation however, these functions do nothing useful.
 * Therefore their respective destroy functions are not used. */

void circ_buff_destroy(circ_buff_t *new_buff){
   pthread_mutex_lock(&(new_buff->circ_read_mut));
   pthread_mutex_lock(&(new_buff->circ_write_mut));
   free(new_buff->buff_ptr);
   pthread_mutex_unlock(&(new_buff->circ_read_mut));
   pthread_mutex_unlock(&(new_buff->circ_write_mut));
   free(new_buff);
   
}





int circ_buff_init_thread(circ_buff_t *new_buff, long *thread_read_pnt,
	                                 unsigned long long *thread_byte_id,
                                     struct timespec *timeout){

   int identical;


   identical=0;
   while(!identical){
      (*thread_byte_id)=new_buff->byte_id;
      (*thread_read_pnt)=new_buff->write_pnt;
      if((*thread_byte_id)==new_buff->byte_id)
         identical=1;
   }

   if((circ_buff_wait_to_start(new_buff,(*thread_byte_id),timeout)) < 0){
      return -1; 
   }
   fprintf(stderr,"*******************************\n");
         fprintf(stderr,"global byte id=%lld\n",new_buff->byte_id);
         fprintf(stderr,"thread byte id=%lld\n",*thread_byte_id);
         fprintf(stderr,"buff_size=%ld\n\n",new_buff->buff_size);
   
   return 0;

}



int circ_buff_wait_to_start(circ_buff_t *new_buff,
	                                 unsigned long long thread_byte_id,
                                     struct timespec *timeout){
   long half_buff;
   int retcode=0;
   struct timeval now;
   struct timespec time_to_wake;
   struct timezone timez;
   half_buff=new_buff->buff_size/2;
   
   while(thread_byte_id <= new_buff->byte_id && 
		   new_buff->byte_id < (thread_byte_id+half_buff)
		   && retcode != ETIMEDOUT){ 
	   
      pthread_mutex_lock(&(new_buff->circ_write_cond_mut));
      gettimeofday(&now,&timez);
      time_to_wake.tv_sec=now.tv_sec + timeout->tv_sec;
   
      time_to_wake.tv_nsec=(now.tv_usec * 1000) + timeout->tv_nsec; 
   
      if(time_to_wake.tv_nsec >= 1000000000){
         time_to_wake.tv_nsec=time_to_wake.tv_nsec % 1000000000;
         time_to_wake.tv_sec++;
      }
      retcode=pthread_cond_timedwait(&(new_buff->circ_write_cond_var),
                                       &(new_buff->circ_write_cond_mut),
	                 			       &time_to_wake);
      pthread_mutex_unlock(&(new_buff->circ_write_cond_mut));
	   
	   
   }
  
   if(retcode==ETIMEDOUT)
      return -1;

   return 0; 

}
/**
 * 
 * 
 *  \warning timeout.tv_sec cannot exceed the maximum value of a long integer
 *           - the current number of seconds since 00:00:00 GMT January 1, 1970.
 * 
 */

int circ_buff_wait_for_read(circ_buff_t *new_buff, struct timespec *timeout){
   
   int retcode=0;
   struct timeval now;
   struct timespec time_to_wake;
   struct timezone timez;
   

   gettimeofday(&now,&timez);
   time_to_wake.tv_sec=now.tv_sec + timeout->tv_sec;
   
   time_to_wake.tv_nsec=(now.tv_usec * 1000) + timeout->tv_nsec; 
   
   if(time_to_wake.tv_nsec >= 1000000000){
      time_to_wake.tv_nsec=time_to_wake.tv_nsec % 1000000000;
      time_to_wake.tv_sec++;
   }
   
   pthread_mutex_lock(&(new_buff->circ_read_cond_mut));
   
   retcode=pthread_cond_timedwait(&(new_buff->circ_read_cond_var),
                                       &(new_buff->circ_read_cond_mut),
				       &time_to_wake);
   pthread_mutex_unlock(&(new_buff->circ_read_cond_mut));
	   
	   
   if(retcode==ETIMEDOUT)
      return -1;

   return 0; 

}





int circ_buff_wait_for_write(circ_buff_t *new_buff, struct timespec *timeout){
   
   int retcode=0;
   struct timeval now;
   struct timespec time_to_wake;
   struct timezone timez;
   
   gettimeofday(&now,&timez);
   time_to_wake.tv_sec=now.tv_sec + timeout->tv_sec;
   
   time_to_wake.tv_nsec=(now.tv_usec * 1000) + timeout->tv_nsec; 
   
   if(time_to_wake.tv_nsec >= 1000000000){
      time_to_wake.tv_nsec=time_to_wake.tv_nsec % 1000000000;
      time_to_wake.tv_sec++;
   }
   
   pthread_mutex_lock(&(new_buff->circ_write_cond_mut));
   
   retcode=pthread_cond_timedwait(&(new_buff->circ_write_cond_var),
                                       &(new_buff->circ_write_cond_mut),
                     &time_to_wake);
   pthread_mutex_unlock(&(new_buff->circ_write_cond_mut));                  
                        
   if(retcode==ETIMEDOUT){
      return -1;
   }

   return 0; 

}




/* If the buffer is resized while other threads are doing a
 * read, those threads will have to reset their read pointer
 * appropriately.  This will most likely cause interuption
 * in the bit stream they are reading from. */

int circ_buff_resize(circ_buff_t *new_buff, long new_size,
                                            struct timespec *timeout){

   unsigned char *resized_buff;
   int ret_val;
   

   free(new_buff->buff_ptr);


   pthread_mutex_lock(&(new_buff->circ_read_mut));
   pthread_mutex_lock(&(new_buff->circ_write_mut));

   if(new_size < new_buff->byte_count){
      fprintf(stderr,"resize_circ_buff: Cannot be resized because there is");
      fprintf(stderr," not enough space for data.\n");
      return -1;
   }
   resized_buff=(unsigned char *)malloc(new_size);
   if(resized_buff == NULL){
      fprintf(stderr,"resize_circ_buff: Cannot be resized,");
      fprintf(stderr," could not allocate memory.\n");
      return -1;
   }
   if((ret_val=circ_buff_read(new_buff,resized_buff,
				   new_buff->byte_count, timeout))>0){
      fprintf(stderr,"resize_circ_buff: Could not read from");
      fprintf(stderr," old buffer during resize\n");
      return -1;
   }
   new_buff->buff_ptr=resized_buff;
   new_buff->write_pnt=new_buff->byte_count;
   new_buff->read_pnt=0;
   new_buff->byte_count=ret_val;
   new_buff->buff_size=new_size;
   //The byte_id is changed to make the reading threads fail
   new_buff->byte_id=new_buff->byte_id + new_buff->buff_size + 1;

   free(new_buff->buff_ptr);
   pthread_mutex_unlock(&(new_buff->circ_read_mut));
   pthread_mutex_unlock(&(new_buff->circ_write_mut));

   return 0;
}


/* This is a protected read that cannot be executed by multiple
 * threads at the same time. */

long circ_buff_read(circ_buff_t *new_buff, void *read_data,
	                                    long bytes_to_read,
                                        struct timespec *timeout){

   long actual_read;
   struct timeval now;
   struct timespec time_to_wake;
   struct timezone timez;
   int retcode=0;
   
   timez.tz_minuteswest=0;

   if(bytes_to_read==0){
      return 0;
   }
   
   pthread_mutex_lock(&(new_buff->circ_read_mut));
   
   //perror("Master Reader is Reading");
   
   while(bytes_to_read > new_buff->byte_count && retcode!=ETIMEDOUT){ 
	 
      pthread_mutex_lock(&(new_buff->circ_write_cond_mut));
      gettimeofday(&now,&timez);
      time_to_wake.tv_sec=now.tv_sec + timeout->tv_sec;
   
      time_to_wake.tv_nsec=(now.tv_usec * 1000) + timeout->tv_nsec; 
   
      if(time_to_wake.tv_nsec >= 1000000000){
         time_to_wake.tv_nsec=time_to_wake.tv_nsec % 1000000000;
         time_to_wake.tv_sec++;
      }
   
      retcode=pthread_cond_timedwait(&(new_buff->circ_write_cond_var),
                                    &(new_buff->circ_write_cond_mut),
	                            &time_to_wake);
      pthread_mutex_unlock(&(new_buff->circ_write_cond_mut));
    
   } 
   
   if(retcode==ETIMEDOUT){
      //fprintf(stderr,"read_circ_buff: Timed out waiting");
      //fprintf(stderr," for the write condition variable.\n");
      pthread_mutex_unlock(&(new_buff->circ_read_mut));
      //perror("Master Reader is Finished Reading: Timeout");
      return 0;
   }
   
   actual_read=bytes_to_read;
   
   //if the bytes wrap
   if(new_buff->read_pnt+actual_read > new_buff->buff_size){
      //perror("Master Reader Bytes Wrapped");
      
      mempcpy(mempcpy(read_data,new_buff->buff_ptr+new_buff->read_pnt,
                       new_buff->buff_size-new_buff->read_pnt),
                       new_buff->buff_ptr,
           actual_read-(new_buff->buff_size-new_buff->read_pnt));
      
   
      new_buff->read_pnt=actual_read-(new_buff->buff_size-new_buff->read_pnt);
   }
   else{
          //perror("Master Reader Bytes did not Wrap");

      mempcpy(read_data,
		      new_buff->buff_ptr+new_buff->read_pnt,actual_read);

      new_buff->read_pnt=(new_buff->read_pnt+actual_read)%new_buff->buff_size;
   }

   pthread_mutex_lock(&(new_buff->circ_size_var_mut));
   new_buff->byte_count=new_buff->byte_count-actual_read;
   pthread_mutex_unlock(&(new_buff->circ_size_var_mut));

   pthread_mutex_unlock(&(new_buff->circ_read_mut));
   
   pthread_mutex_lock(&(new_buff->circ_read_cond_mut));
   pthread_cond_broadcast(&(new_buff->circ_read_cond_var)); 
   pthread_mutex_unlock(&(new_buff->circ_read_cond_mut));

   
   return actual_read;
}





long circ_buff_read_thread(circ_buff_t *new_buff, void *read_data,
	       		long *thread_read_pnt,
		        unsigned long long *thread_byte_id,
			    long read_size,
                struct timespec *timeout){

   long actual_read=0;
   struct timeval now;
   struct timespec time_to_wake;
   struct timezone timez;
   int retcode=0;

   
   timez.tz_minuteswest=0;
   if(read_size==0){
      return 0;
   }
   while(actual_read==0 && retcode!=ETIMEDOUT){
      /* Note that this works even if the byte_id wraps around to 1 
       * because byte_id is an unsigned int */
      if(new_buff->byte_id-(*thread_byte_id) >= new_buff->buff_size){
         fprintf(stderr,"global byte id=%lld\n",new_buff->byte_id);
         fprintf(stderr,"thread byte id=%lld\n",*thread_byte_id);
         fprintf(stderr,"buff_size=%ld\n",new_buff->buff_size);

         fprintf(stderr,"read_thread_circ_buff: Overrun in the");
         fprintf(stderr," buffer (not reading fast enough).\n");
         return -1;
      }
      if(new_buff->byte_id-(*thread_byte_id)<read_size){
         actual_read=new_buff->byte_id-(*thread_byte_id);
      }
      else{
         actual_read=read_size; 
      }
      if(actual_read==0){
         //fprintf(stderr,"read_thread_circ_buff: Waiting for a write.\n");
         pthread_mutex_lock(&(new_buff->circ_write_cond_mut));
         gettimeofday(&now,&timez);
         time_to_wake.tv_sec=now.tv_sec + timeout->tv_sec;
   
         time_to_wake.tv_nsec=(now.tv_usec * 1000) + timeout->tv_nsec; 
   
         if(time_to_wake.tv_nsec >= 1000000000){
            time_to_wake.tv_nsec=time_to_wake.tv_nsec % 1000000000;
            time_to_wake.tv_sec++;
         }
         retcode=pthread_cond_timedwait(&(new_buff->circ_write_cond_var),
                                       &(new_buff->circ_write_cond_mut),
				       &time_to_wake);
         pthread_mutex_unlock(&(new_buff->circ_write_cond_mut));
         
      }
   
   }

   if(actual_read==0){
      //fprintf(stderr,"read_thread_circ_buff: Timed out waiting");
      //fprintf(stderr," for the write condition variable.\n");
      return 0;
   }

   
   if(*thread_read_pnt >= new_buff->buff_size){
      fprintf(stderr,"read_thread_circ_buff: Pointer out of range");
      fprintf(stderr," buffer must have been resized.\n");
      return -1;
   }

   if(new_buff->byte_id-(*thread_byte_id) >= new_buff->buff_size){
      fprintf(stderr,"global byte id=%lld\n",new_buff->byte_id);
      fprintf(stderr,"thread byte id=%lld\n",*thread_byte_id);
      fprintf(stderr,"buff_size=%ld\n",new_buff->buff_size);

      fprintf(stderr,"read_thread_circ_buff: Overrun in the");
      fprintf(stderr," buffer (not reading fast enough).\n");
      return -1;
   }


   //if the bytes wrap
   if((*thread_read_pnt)+actual_read > new_buff->buff_size){
      
      mempcpy(mempcpy(read_data, new_buff->buff_ptr+(*thread_read_pnt),
              new_buff->buff_size-(*thread_read_pnt)),
		      new_buff->buff_ptr,
		      actual_read-(new_buff->buff_size-(*thread_read_pnt)));
      
      (*thread_read_pnt)=actual_read-(new_buff->buff_size-(*thread_read_pnt));
   }
   else{
      mempcpy(read_data,
		      new_buff->buff_ptr+(*thread_read_pnt),actual_read);
      
      (*thread_read_pnt)=((*thread_read_pnt)+actual_read)%new_buff->buff_size;
   }

   if(new_buff->byte_id-(*thread_byte_id) > new_buff->buff_size){
      fprintf(stderr,"read_thread_circ_buff: Overrun in the buffer");
      fprintf(stderr," (occured during function call).\n");
      return -1;
   }

   (*thread_byte_id)+=actual_read;

   return actual_read;
}


/* This is a protected write that cannot be executed by multiple threads at
 * once.
 */

long circ_buff_write(circ_buff_t *new_buff, void *write_data,
	                                 	long bytes_to_write,
                                        struct timespec *timeout){

   struct timeval now;
   struct timespec time_to_wake;
   struct timezone timez;
   int retcode=0;
   long actual_write=0;

   timez.tz_minuteswest=0;

   if(bytes_to_write==0){
      return 0;
   }

   pthread_mutex_lock(&(new_buff->circ_write_mut));
   
   

   while(actual_write==0 && retcode!=ETIMEDOUT){
      if(bytes_to_write+new_buff->byte_count > new_buff->buff_size)
         actual_write=new_buff->buff_size-new_buff->byte_count;
      else
         actual_write=bytes_to_write; 

      if(actual_write==0){
         pthread_mutex_lock(&(new_buff->circ_read_cond_mut));
         gettimeofday(&now,&timez);
         time_to_wake.tv_sec=now.tv_sec + timeout->tv_sec;
   
         time_to_wake.tv_nsec=(now.tv_usec * 1000) + timeout->tv_nsec; 
   
         if(time_to_wake.tv_nsec >= 1000000000){
            time_to_wake.tv_nsec=time_to_wake.tv_nsec % 1000000000;
            time_to_wake.tv_sec++;
         }
         retcode=pthread_cond_timedwait(&(new_buff->circ_read_cond_var),
                                       &(new_buff->circ_read_cond_mut),
				       &time_to_wake);
         pthread_mutex_unlock(&(new_buff->circ_read_cond_mut));
      }
   
   }
   if(actual_write==0){
      fprintf(stderr,"write_circ_buff: Timed out waiting");
      fprintf(stderr," for the read condition variable.\n");
         perror("Master Writer is finished writing timed out");
      pthread_mutex_unlock(&(new_buff->circ_write_mut));
      return 0;
   }
   

   //if write will wrap
   if(new_buff->write_pnt+actual_write > new_buff->buff_size){
      memcpy(new_buff->buff_ptr+new_buff->write_pnt,
		      write_data,new_buff->buff_size-new_buff->write_pnt);

      memcpy(new_buff->buff_ptr,
		      write_data+(new_buff->buff_size-new_buff->write_pnt),
		      actual_write-
		      (new_buff->buff_size-new_buff->write_pnt));
      

      new_buff->write_pnt=actual_write-
	      (new_buff->buff_size-new_buff->write_pnt);
   }
   else{
      memcpy(new_buff->buff_ptr+new_buff->write_pnt,
		      write_data,actual_write);
      

      new_buff->write_pnt=(new_buff->write_pnt+actual_write)%
	                                         new_buff->buff_size;
   }

   new_buff->byte_id+=actual_write;

   pthread_mutex_lock(&(new_buff->circ_size_var_mut));
   new_buff->byte_count+=actual_write;
   pthread_mutex_unlock(&(new_buff->circ_size_var_mut));

   pthread_mutex_unlock(&(new_buff->circ_write_mut));
   
   pthread_mutex_lock(&(new_buff->circ_write_cond_mut));
   pthread_cond_broadcast(&(new_buff->circ_write_cond_var)); 
   pthread_mutex_unlock(&(new_buff->circ_write_cond_mut));


   return actual_write;
}


/** @}*/

