/*! \file alsad_shared_lib.c
 *  \brief This is the implementation of the Shared Library
 *  \author Daniel R. Warren
 *  \version 1.0
 *  \date    November 2004
 *  \ingroup shared_lib
*/


 /** \defgroup shared_lib The Shared Library
 *   \ingroup application_components
 *   \brief Library that provides common tasks to the client and alsad.
 * 
   
 */
 
/** \addtogroup shared_lib */
/** @{*/





#include "alsad_shared_lib.h"



int alsad_config_key_load(char *config_file, alsad_config_keys_t *key_struct){

   int loaded_pairs;
   key_value_pair_t *config_pairs;
   char *tmp_username; 
   char *tmp_priv_key_file;
   char *tmp_pub_key_dir;
   char *tmp_max_stream_sinks;
   char *tmp_pcm_format;
   char *tmp_channels;
   char *tmp_open_mode;
   char *tmp_circ_buff_size;
   char *tmp_max_circ_buff_size;
   char *tmp_card_buf_time;
   char *tmp_card_period_time;
   char *tmp_card_device_name;
   char *tmp_port;



   config_pairs=config_file_read(config_file,&loaded_pairs);
   if(config_pairs==NULL){
      return -1; 
   }

   

   tmp_username=config_file_get_value(config_pairs,
                        ALSAD_KEY_USERNAME,loaded_pairs);
   if(tmp_username==NULL){
      key_struct->config_username=
	      (char *)malloc(sizeof(char)*strlen(ALSAD_DFLT_USERNAME)+1);
      strcpy(key_struct->config_username,ALSAD_DFLT_USERNAME);
   }
   else{
      key_struct->config_username=
	      (char *)malloc(sizeof(char)*strlen(tmp_username)+1);
      strcpy(key_struct->config_username,tmp_username);
   }

   tmp_priv_key_file=config_file_get_value(config_pairs,
		                        ALSAD_KEY_PRIV_KEY_FILE,loaded_pairs);
   if(tmp_priv_key_file==NULL){
      key_struct->config_priv_key_file=(char *)malloc(sizeof(char)*
		                 strlen(ALSAD_DFLT_PRIV_KEY_FILE)+1);
      strcpy(key_struct->config_priv_key_file,ALSAD_DFLT_PRIV_KEY_FILE);
   }
   else{
      key_struct->config_priv_key_file=(char *)malloc(sizeof(char)*
		                        strlen(tmp_priv_key_file)+1);
      strcpy(key_struct->config_priv_key_file,tmp_priv_key_file);
   }



   tmp_pub_key_dir=config_file_get_value(config_pairs,
		                        ALSAD_KEY_PUB_KEY_DIR,loaded_pairs);
   if(tmp_pub_key_dir==NULL){
      key_struct->config_pub_key_dir=(char *)malloc(sizeof(char)*
		                        strlen(ALSAD_DFLT_PUB_KEY_DIR)+1);
      strcpy(key_struct->config_pub_key_dir, ALSAD_DFLT_PUB_KEY_DIR);
   }
   else{
      key_struct->config_pub_key_dir=(char *)malloc(sizeof(char)*
		                        strlen(tmp_pub_key_dir)+1);
      strcpy(key_struct->config_pub_key_dir,tmp_pub_key_dir);
   }

   

   tmp_max_stream_sinks=config_file_get_value(config_pairs,
		                        ALSAD_KEY_MAX_STREAM_SINKS,loaded_pairs);
   if(tmp_max_stream_sinks==NULL){
      key_struct->config_max_stream_sinks=(char *)malloc(sizeof(char)*
		                        strlen(ALSAD_DFLT_MAX_STREAM_SINKS)+1);
      strcpy(key_struct->config_max_stream_sinks,ALSAD_DFLT_MAX_STREAM_SINKS);
   }
   else{
      key_struct->config_max_stream_sinks=(char *)malloc(sizeof(char)*
                                           strlen(tmp_max_stream_sinks)+1);
      strcpy(key_struct->config_max_stream_sinks,tmp_max_stream_sinks);
   }



   tmp_pcm_format=config_file_get_value(config_pairs,
		                    ALSAD_KEY_PCM_FORMAT,loaded_pairs);
   if(tmp_pcm_format==NULL){ 
      key_struct->config_pcm_format=(char *)malloc(sizeof(char)*
		                        strlen(ALSAD_DFLT_PCM_FORMAT)+1);
      strcpy(key_struct->config_pcm_format,ALSAD_DFLT_PCM_FORMAT);
   }
   else{
      key_struct->config_pcm_format=(char *)malloc(sizeof(char)*
                                           strlen(tmp_pcm_format)+1);
      strcpy(key_struct->config_pcm_format,tmp_pcm_format);
   }
      
	   
	   
   tmp_channels=config_file_get_value(config_pairs,
		                    ALSAD_KEY_CHANNELS,loaded_pairs);
   if(tmp_channels==NULL){ 
      key_struct->config_channels=(char *)malloc(sizeof(char)*
		                        strlen(ALSAD_DFLT_CHANNELS)+1);
      strcpy(key_struct->config_channels,ALSAD_DFLT_CHANNELS);
   }
   else{
      key_struct->config_channels=(char *)malloc(sizeof(char)*
                                           strlen(tmp_channels)+1);
      strcpy(key_struct->config_channels,tmp_channels);
   }
      
   
   
   tmp_open_mode=config_file_get_value(config_pairs,
		                    ALSAD_KEY_OPEN_MODE,loaded_pairs);
   if(tmp_open_mode==NULL){ 
      key_struct->config_open_mode=(char *)malloc(sizeof(char)*
		                        strlen(ALSAD_DFLT_OPEN_MODE)+1);
      strcpy(key_struct->config_open_mode,ALSAD_DFLT_OPEN_MODE);
   }
   else{
      key_struct->config_open_mode=(char *)malloc(sizeof(char)*
                                           strlen(tmp_open_mode)+1);
      strcpy(key_struct->config_open_mode,tmp_open_mode);
   }
      
   
   tmp_circ_buff_size=config_file_get_value(config_pairs,
                           ALSAD_KEY_CIRC_BUFF_SIZE,loaded_pairs);
   if(tmp_circ_buff_size==NULL){ 
      key_struct->config_circ_buff_size=(char *)malloc(sizeof(char)*
                                strlen(ALSAD_DFLT_CIRC_BUFF_SIZE)+1);
      strcpy(key_struct->config_circ_buff_size,ALSAD_DFLT_CIRC_BUFF_SIZE);
   }
   else{
      key_struct->config_circ_buff_size=(char *)malloc(sizeof(char)*
                                           strlen(tmp_circ_buff_size)+1);
      strcpy(key_struct->config_circ_buff_size,tmp_circ_buff_size);
   }
   
   
   tmp_max_circ_buff_size=config_file_get_value(config_pairs,
                           ALSAD_KEY_MAX_CIRC_BUFF_SIZE,loaded_pairs);
   if(tmp_max_circ_buff_size==NULL){ 
      key_struct->config_max_circ_buff_size=(char *)malloc(sizeof(char)*
                                strlen(ALSAD_DFLT_MAX_CIRC_BUFF_SIZE)+1);
      strcpy(key_struct->config_max_circ_buff_size,
                                        ALSAD_DFLT_MAX_CIRC_BUFF_SIZE);
   }
   else{
      key_struct->config_max_circ_buff_size=(char *)malloc(sizeof(char)*
                                        strlen(tmp_max_circ_buff_size)+1);
      strcpy(key_struct->config_max_circ_buff_size,tmp_max_circ_buff_size);
   }
   
   
   

   tmp_card_buf_time=config_file_get_value(config_pairs,
		                    ALSAD_KEY_CARD_BUF_TIME,loaded_pairs);
   if(tmp_card_buf_time==NULL){ 
      key_struct->config_card_buf_time=(char *)malloc(sizeof(char)*
		                        strlen(ALSAD_DFLT_CARD_BUF_TIME)+1);
      strcpy(key_struct->config_card_buf_time,ALSAD_DFLT_CARD_BUF_TIME);
   }
   else{
      key_struct->config_card_buf_time=(char *)malloc(sizeof(char)*
                                           strlen(tmp_card_buf_time)+1);
      strcpy(key_struct->config_card_buf_time,tmp_card_buf_time);
   }
      

    
   tmp_card_period_time=config_file_get_value(config_pairs,
		                    ALSAD_KEY_CARD_PERIOD_TIME,loaded_pairs);
   if(tmp_card_period_time==NULL){ 
      key_struct->config_card_period_time=(char *)malloc(sizeof(char)*
		                        strlen(ALSAD_DFLT_CARD_PERIOD_TIME)+1);
      strcpy(key_struct->config_card_period_time,ALSAD_DFLT_CARD_PERIOD_TIME);
   }
   else{
      key_struct->config_card_period_time=(char *)malloc(sizeof(char)*
                                           strlen(tmp_card_period_time)+1);
      strcpy(key_struct->config_card_period_time,tmp_card_period_time);
   }
      
  

   tmp_card_device_name=config_file_get_value(config_pairs,
		                    ALSAD_KEY_CARD_DEVICE_NAME,loaded_pairs);
   if(tmp_card_device_name==NULL){ 
      key_struct->config_card_device_name=(char *)malloc(sizeof(char)*
		                        strlen(ALSAD_DFLT_CARD_DEVICE_NAME)+1);
      strcpy(key_struct->config_card_device_name,ALSAD_DFLT_CARD_DEVICE_NAME);
   }
   else{
      key_struct->config_card_device_name=(char *)malloc(sizeof(char)*
                                           strlen(tmp_card_device_name)+1);
      strcpy(key_struct->config_card_device_name,tmp_card_device_name);
   }
      
 

   tmp_port=config_file_get_value(config_pairs,
		                    ALSAD_KEY_PORT,loaded_pairs);
   if(tmp_port==NULL){ 
      key_struct->config_port=(char *)malloc(sizeof(char)*
		                        strlen(ALSAD_DFLT_PORT)+1);
      strcpy(key_struct->config_port,ALSAD_DFLT_PORT);
   }
   else{
      key_struct->config_port=(char *)malloc(sizeof(char)*
                                           strlen(tmp_port)+1);
      strcpy(key_struct->config_port,tmp_port);
   }
      

   config_file_destroy(config_pairs,loaded_pairs);

   return loaded_pairs;
}



int alsad_conn_with_host(alsad_connect_addr_t *conn_addr,
		alsad_net_structs_t *send_structs,
		char *userid, char *priv_key_file, 
                alsad_control_t *conn_ctrl){

   int retval;
   struct in_addr serv_addr;
   struct hostent *server_hostent;
   int server_sock_desc;
   alsad_control_t rcvd_control;
   char error_buf[ALSAD_MAX_ERR_SIZE];


   server_hostent=gethostbyname(conn_addr->hostname);
   if(server_hostent == NULL){
      fprintf(stderr,"alsad_conn_with_host: gethostbyname failed\n");
      return -1;
   }
   memcpy(&serv_addr, server_hostent->h_addr_list[0], sizeof(struct in_addr));

   server_sock_desc=create_connect(&serv_addr, (short)conn_addr->port);
   if(server_sock_desc <= 0){
      fprintf(stderr,"alsad_conn_with_host: Could not create a connection\n");
      return -1;
   }
/** \todo Make authentication an option
 */
   retval=auth_client(server_sock_desc, userid, priv_key_file);
   if(retval<0){
      fprintf(stderr,"alsad_conn_with_host: Authentication failed\n");
      close(server_sock_desc);
      return -1;
   }

   if((retval=alsad_send_control(server_sock_desc, conn_ctrl)) <= 0){
      fprintf(stderr,"alsad_conn_with_host: Could not send control message\n");
      close(server_sock_desc);
      return -1;
   }

   if((retval=alsad_reply_structs(server_sock_desc,send_structs)) < 0){
      fprintf(stderr,"alsad_conn_with_host: Not enough information supplied\n");
      close(server_sock_desc);
      return -1;
   }
    
   if((retval=alsad_recv_control(server_sock_desc, &rcvd_control)) <= 0){
      fprintf(stderr,"alsad_conn_with_host: Could not recv control message\n");
      close(server_sock_desc);
      return -1;
   }
   /*! \todo Make the control struct contain a error number so that the server
    *  can provide more information if there is an error.
    */
   if((rcvd_control.ctrl & ALSAD_MSG_RQST_ERR)==ALSAD_MSG_RQST_ERR){ 
      fprintf(stderr,"alsad_conn_with_host: Server denied request\n");
      retval=alsad_strerror(rcvd_control.code, error_buf, ALSAD_MAX_ERR_SIZE);
      if(retval<0){
        fprintf(stderr,"alsad_conn_with_host: Undefined error\n");
      }
      else{
        fprintf(stderr,"alsad_conn_with_host: %s\n",error_buf);
      }
      close(server_sock_desc);
      return -1;
   }
   
   if((rcvd_control.ctrl & ALSAD_MSG_RQST_OK)!=ALSAD_MSG_RQST_OK){ 
      fprintf(stdout,"alsad_conn_with_host: Message not OK or ERR\n");
      close(server_sock_desc);
      return -1;
   }

   return server_sock_desc; 
}


int alsad_handle_client_protocol(int sock, int handle, alsad_control_t *conn_ctrl){

   int retval;
   ssize_t long_ret;
   int print_sink; 
   alsad_control_t rcvd_control;
   alsad_control_t send_control;
   alsad_data_pipe_t rcvd_data_pipe;
   alsad_hw_params_t rcvd_hw_params;
   alsad_sw_params_t rcvd_sw_params;
   alsad_connect_addr_t rcvd_conn_addr;
   alsad_audio_hdr_t send_audio_hdr;
   alsad_audio_hdr_t rcvd_audio_hdr;
   char databuf[ALSAD_MAX_TRANS_SIZE];
   ssize_t rcvd_bytes;
   
   //alsad_control_t send_control;
   //alsad_audio_hdr_t rcvd_audio_hdr, send_audio_hdr;
   alsad_stream_props_t rcvd_stream_props;
   //alsad_connect_addr_t local_conn_addr;
   //unsigned long bits_per_frame;
   //unsigned long period_size_in_bytes;
   //unsigned long long long_size_in_bytes;
   //int matched;
   //struct timespec wait_time;
   //struct sockaddr_in localaddress;
   //char *audiobuf=NULL;
   //int session_terminated=0;
   //int len_sockaddr=sizeof(struct sockaddr);


   rcvd_bytes=0;

   if((conn_ctrl->ctrl & ALSAD_RQST_ADD_SRC)==ALSAD_RQST_ADD_SRC && 
      (conn_ctrl->ctrl & ALSAD_PATH_CUR_SOCK)==ALSAD_PATH_CUR_SOCK){ 

      if((retval=alsad_recv_control(sock, &rcvd_control)) <= 0){
         fprintf(stderr,"alsad_handle_client_protocol: Could not recv control message\n");
         close(sock);
         return -1;
      }
   
      /*! \todo I need to set up the client so that it ends cleanly
       */
       
      while(1){ 
     
         if((rcvd_control.ctrl & ALSAD_SEND_AUDIO_BUF)!=
			                      ALSAD_SEND_AUDIO_BUF){
            
            if((rcvd_control.ctrl & ALSAD_MSG_CLOSE_CONN) ==
			                      ALSAD_MSG_CLOSE_CONN){
               fprintf(stderr,"alsad_handle_client_protocol: Closing Connection Upon Request\n");
	         }
            else{
               fprintf(stderr,"alsad_handle_client_protocol: Unexpected control message\n");
    	      }   
            close(sock);
            return -1;
         }
        
         if(rcvd_control.code > ALSAD_MAX_TRANS_SIZE){
            fprintf(stderr,"alsad_handle_client_protocol: Requested larger then MAX_TRANS \n");
            close(sock);
            return -1;
         }


	     //Note that alsad actually sends data after every read function regardless
	     //of whether all the data has been read.  The read_write_function guarantees
	     //that all of the data has been read.
         if((long_ret=alsad_file_read(handle,databuf,
	                                (size_t)rcvd_control.code)) < 0){
            fprintf(stderr,"alsad_handle_client_protocol: Could not read from channel or file\n");
            close(sock);
            return -1;
         }

         if(long_ret<0){
            fprintf(stderr,"alsad_handle_client_protocol: Error Reading from file\n");
            close(sock);
            return -1;
         }

         send_audio_hdr.payload_size=(unsigned int)long_ret;
         if((retval=alsad_send_audio_hdr(sock,
	                              &send_audio_hdr, databuf)) <= 0){
            fprintf(stderr,"alsad_handle_client_protocol: ");
            fprintf(stderr,"Could not send audio header\n");
            close(sock);
            return -1;
         }

         if(long_ret!=rcvd_control.code){
            fprintf(stderr,"alsad_handle_client_protocol: End of file reached\n");
            close(sock);
            return 0;
         }


         fprintf(stdout,"Receiving rcvd_control.code %u\n", send_audio_hdr.payload_size);
         if((retval=alsad_recv_control(sock, &rcvd_control)) <= 0){
            fprintf(stderr,"alsad_handle_client_protocol: Could not recv control message\n");
            close(sock);
            return -1;
         }
 
      } 

   }

   if((conn_ctrl->ctrl & ALSAD_RQST_ADD_SINK)==ALSAD_RQST_ADD_SINK && 
      (conn_ctrl->ctrl & ALSAD_PATH_CUR_SOCK)==ALSAD_PATH_CUR_SOCK){ 

      while(1){ //I need to set up the client so that it ends cleanly
    
         send_control.ctrl=ALSAD_SEND_AUDIO_BUF;
         send_control.code=ALSAD_MAX_TRANS_SIZE; 
         fprintf(stderr,"alsad_handle_client_protocol: sending control message\n");
         
         if((retval=alsad_send_control(sock, &send_control)) <= 0){
            fprintf(stderr,"alsad_handle_client_protocol: Could not send control message\n");
            close(sock);
            return -1;
         }
         
         rcvd_bytes=0;
         // Loop until the client has recieved all the data it requested
         while(rcvd_bytes < ALSAD_MAX_TRANS_SIZE){ 
            fprintf(stdout,"alsad_handle_client_protocol: receiving control message\n");
            if((retval=alsad_recv_control(sock, &rcvd_control)) <= 0){
               fprintf(stderr,"alsad_handle_client_protocol: Could not recv control message\n");
               close(sock);
               return -1;
            }
            
            if((rcvd_control.ctrl & ALSAD_RECV_AUDIO_BUF)!=
                               ALSAD_RECV_AUDIO_BUF){
               if((rcvd_control.ctrl & ALSAD_MSG_KEEP_ALIVE) ==
                                 ALSAD_MSG_KEEP_ALIVE){
                  continue;
               }
               if((rcvd_control.ctrl & ALSAD_MSG_CLOSE_CONN) ==
                                 ALSAD_MSG_CLOSE_CONN){
                  fprintf(stderr,"alsad_handle_client_protocol: Closing Connection Upon Request\n");
               }
               else{
                  fprintf(stderr,"alsad_handle_client_protocol: Unexpected control message\n");
               }
               close(sock);
               return -1;
            }
             
            if((retval=alsad_recv_audio_hdr(sock,
            			               &rcvd_audio_hdr))<=0){
               fprintf(stderr,"alsad_handle_client_protocol: ");
               fprintf(stderr,"Unable to recv_audio_hdr\n");
               close(sock);
               return -1;
            }
            
            if(rcvd_audio_hdr.payload_size+rcvd_bytes > ALSAD_MAX_TRANS_SIZE){
               fprintf(stderr,"alsad_handle_client_protocol: ");
               fprintf(stderr,"Sent more data than requested\n");
               close(sock);
               return -1;
            }
            fprintf(stderr,"Receiving audio payload\n");
            if((retval=safe_sock_recv(sock, databuf+rcvd_bytes,
            			   rcvd_audio_hdr.payload_size,
                              ALSAD_GENERIC_SOCK_TIMEOUT))<=0){
               fprintf(stderr,"alsad_handle_client_protocol: ");
               fprintf(stderr,"Unable to receive audio data\n");
               close(sock);
               return -1;
            }
            
            fprintf(stderr,"Writing to file\n");
            if((long_ret=alsad_file_write(handle,databuf+rcvd_bytes,
            	    (size_t)rcvd_audio_hdr.payload_size))<=0){
               fprintf(stderr,"alsad_handle_client_protocol: ");
               fprintf(stderr,"Unable to write data to file\n");
               close(sock);
               return -1;
            }
            
            rcvd_bytes+=long_ret;
         }
            
      } 
         
         
         
   }

   if((conn_ctrl->ctrl & ALSAD_RQST_LIST_STREAM) == ALSAD_RQST_LIST_STREAM){

      if((retval=alsad_recv_control(sock, &rcvd_control)) <= 0){
         fprintf(stderr,"alsad_handle_client_protocol: Could not recv control message\n");
         close(sock);
         return -1;
      }
   
      fprintf(stdout,"\nChannel ID\tFormat\t   Channels\tRate\tText Description\n");
      fprintf(stdout,"------------------------------------------------");
      fprintf(stdout,"----------------------------\n");
      while((rcvd_control.ctrl & ALSAD_MSG_RQST_ERR)!=ALSAD_MSG_RQST_ERR && 
	    (rcvd_control.ctrl & ALSAD_MSG_RQST_OK)!=ALSAD_MSG_RQST_OK){
	 
         if((rcvd_control.ctrl & ALSAD_RECV_STREAM_PROPS)==
			                            ALSAD_RECV_STREAM_PROPS){
            if((retval=alsad_recv_stream_props(sock,
					            &rcvd_stream_props))<=0){
               fprintf(stderr,"alsad_handle_client_protocol: ");
	           fprintf(stderr,"Unable to recv_audio_chan\n");
               close(sock);
	           return -1;
            }
            fprintf(stdout,"%d\t\t%s\t   %u\t\t%u\t%s\n",
			   rcvd_stream_props.identifier,
			   snd_pcm_format_name(rcvd_stream_props.format),
			   rcvd_stream_props.channels,
			   rcvd_stream_props.rate,
			   rcvd_stream_props.text_desc);
         }
         else{
            fprintf(stderr,"\nalsad_handle_client_protocol: Unexpected request...\n");
            close(sock);
            return -1;
         }
         if((retval=alsad_recv_control(sock, &rcvd_control)) <= 0){
            fprintf(stderr,"alsad_handle_client_protocol: Could not recv control message\n");
            close(sock);
            return -1;
         }
      }
      
      if((rcvd_control.ctrl & ALSAD_MSG_RQST_ERR)==ALSAD_MSG_RQST_ERR){
         fprintf(stderr,"alsad_handle_client_protocol: Received error message from server\n");
         close(sock);
         return -1;
      }
             
   }


   print_sink=0;
   if((conn_ctrl->ctrl & ALSAD_RQST_LIST_PIPE) == ALSAD_RQST_LIST_PIPE){
   
      if((retval=alsad_recv_control(sock, &rcvd_control)) <= 0){
         fprintf(stderr,"alsad_handle_client_protocol: Could not recv control message\n");
         close(sock);
         return -1;
      }
   
      fprintf(stdout,"\n\tPath ID   PCM Name  Buffer Time  ");
      fprintf(stdout,"Period Time    Host:Port\n");
      fprintf(stdout,"------------------------------------------------");
      fprintf(stdout,"----------------------------\n");
      fprintf(stdout,"Sources");
      while((rcvd_control.ctrl & ALSAD_MSG_RQST_ERR)!=ALSAD_MSG_RQST_ERR && 
            (rcvd_control.ctrl & ALSAD_MSG_RQST_OK)!=ALSAD_MSG_RQST_OK){
         if((rcvd_control.ctrl & ALSAD_RECV_DATA_PIPE)==
		                        ALSAD_RECV_DATA_PIPE &&
			                        rcvd_control.code==1 && !print_sink){
            fprintf(stdout,"\nSinks");
            print_sink=1;
         }
         if((rcvd_control.ctrl & ALSAD_RECV_DATA_PIPE)==
		                            ALSAD_RECV_DATA_PIPE){
            if((retval=alsad_recv_data_pipe(sock,
				            &rcvd_data_pipe))<=0){
               fprintf(stderr,"alsad_handle_client_protocol: ");
               fprintf(stderr,"Unable to recv_data_pipe\n");
               close(sock);
               return -1;
            }
            fprintf(stdout,"\n\t%d\t  ",
            rcvd_data_pipe.identifier);
         }
         else if((rcvd_control.ctrl & ALSAD_RECV_HW_PARAMS)==
		                            ALSAD_RECV_HW_PARAMS){
            if((retval=alsad_recv_hw_params(sock,
				            &rcvd_hw_params))<=0){
               fprintf(stderr,"alsad_handle_client_protocol: ");
               fprintf(stderr,"Unable to recv_hw_params\n");
               close(sock);
               return -1;
            }
            fprintf(stdout,"%s      %u      %u\t",
		      rcvd_hw_params.pcm_name,
		      rcvd_hw_params.buffer_time,
		      rcvd_hw_params.period_time);
         }
         else{
            if((rcvd_control.ctrl & ALSAD_RECV_SW_PARAMS)==
		                            ALSAD_RECV_SW_PARAMS){
               if((retval=alsad_recv_sw_params(sock,
				                    &rcvd_sw_params))<=0){
                  fprintf(stderr,"alsad_handle_client_protocol: ");
                  fprintf(stderr,"Unable to recv_sw_params\n");
                  close(sock);
                  return -1;
               }
               //Print out sw_params info
            }	
            else{
               fprintf(stdout,"\t\t\t\t      ");
               if((rcvd_control.ctrl & ALSAD_RECV_CONN_ADDR)==
                                        ALSAD_RECV_CONN_ADDR){
                  if((retval=alsad_recv_conn_addr(sock,
		                                  &rcvd_conn_addr))<=0){
                     fprintf(stderr,"alsad_handle_client_protocol: ");
                     fprintf(stderr,"Unable to recv_conn_addr\n");
                     close(sock);
                     return -1;
                  }
                  if(strlen(rcvd_conn_addr.hostname)!=0){
                     fprintf(stdout,"%s:%d",
                     rcvd_conn_addr.hostname,
                     rcvd_conn_addr.port);
                  }
               }
               else{
                  fprintf(stderr,"\nalsad_handle_client_protocol: Unexpected request...\n");
                  close(sock);
                  return -1;
               }
            }
         } 
         if((retval=alsad_recv_control(sock, &rcvd_control)) <= 0){;
            fprintf(stderr,"alsad_handle_client_protocol: Could not recv control message\n");
            close(sock);
            return -1;
         }
      }
      if(!print_sink)
         fprintf(stdout,"\nSinks");
   }

   fprintf(stdout,"\n\n");
 
   close(sock);
   return 0;
}





int alsad_configure_snd_card(snd_pcm_t **handle,
		                 alsad_stream_props_t *rcvd_stream_props,
	                        	alsad_hw_params_t *rcvd_hw_params){

   int err;
   unsigned int temp_rate;
   snd_pcm_hw_params_t *params;
   snd_pcm_uframes_t buffer_size_in_frames;
   snd_pcm_uframes_t frames_per_period;

   fprintf(stderr,"Device %s\n", rcvd_hw_params->pcm_name);

   // open pcm open_mode is either 0 or (open_mode |= SND_PCM_NONBLOCK)
   if((err=snd_pcm_open(handle, (char *)rcvd_hw_params->pcm_name,
		   rcvd_hw_params->stream, rcvd_hw_params->open_mode)) < 0){
      fprintf(stderr,"alsad_configure_snd_card: ");
      fprintf(stderr,"Error opening pcm\n");
      return 0;
   }

   snd_pcm_hw_params_alloca(&params);
   if((err=snd_pcm_hw_params_any(*handle, params)) < 0){
      fprintf(stderr,"alsad_configure_snd_card: ");
      fprintf(stderr,"Could not extract hardware parameters\n");
      return 0;
   }

   if((err=snd_pcm_hw_params_set_access(*handle, params,
                                      SND_PCM_ACCESS_RW_INTERLEAVED)) < 0){
      fprintf(stderr,"alsad_configure_snd_card: ");
      fprintf(stderr,"Could not set access as interleaved\n");
      return 0;
   }

   if((err=snd_pcm_hw_params_set_format(*handle, params,
                                             rcvd_stream_props->format)) < 0){
      fprintf(stderr,"alsad_configure_snd_card: ");
      fprintf(stderr,"Could not set format\n");
      return 0;
   }

   if((err=snd_pcm_hw_params_set_channels(*handle, params,
                                           rcvd_stream_props->channels)) < 0){
      fprintf(stderr,"alsad_configure_snd_card: ");
      fprintf(stderr,"Could not set channels\n");
      return 0;
   }

   fprintf(stderr,"Channels %u\n", rcvd_stream_props->channels);

   if((err=snd_pcm_hw_params_get_channels(params,
                                           &rcvd_stream_props->channels)) < 0){
      fprintf(stderr,"alsad_configure_snd_card: ");
      fprintf(stderr,"Could not set channels\n");
      return 0;
   }


   fprintf(stderr,"Set Channels %u\n", rcvd_stream_props->channels);


   temp_rate=rcvd_stream_props->rate;
   if((err=snd_pcm_hw_params_set_rate_near(*handle, params,
                                         &(rcvd_stream_props->rate), 0)) < 0){
      fprintf(stderr,"alsad_configure_snd_card: ");
      fprintf(stderr,"Could not set rate\n");
      return 0;
   }


   if(temp_rate != rcvd_stream_props->rate){
       fprintf(stderr,"alsad_configure_snd_card: ");
       fprintf(stderr,"Rate requested not set\n");
       return 0;
   }
  
  
   /*if((float)temp_rate * 1.05 < rcvd_stream_props->rate ||
                          (float)temp_rate * 0.95 > rcvd_stream_props->rate){
       fprintf(stderr,"alsad_configure_snd_card: ");
       fprintf(stderr,"Rate requested not set\n");
       return 0;
   }*/
   
   
   temp_rate=rcvd_hw_params->buffer_time;
      fprintf(stderr,"Buffer Time %u\n", rcvd_hw_params->buffer_time);
   if((err=snd_pcm_hw_params_set_buffer_time_near(*handle, params,
                                  &(rcvd_hw_params->buffer_time), 0)) < 0){
      fprintf(stderr,"alsad_configure_snd_card: ");
      fprintf(stderr,"Could not set buffer time\n");
      return 0;
   }
   

   fprintf(stderr,"New Buffer Time %u\n", rcvd_hw_params->buffer_time);
   

   if((float)temp_rate * 1.05 < rcvd_hw_params->buffer_time ||
                          (float)temp_rate * 0.95 > rcvd_hw_params->buffer_time){
       fprintf(stderr,"alsad_configure_snd_card: ");
       fprintf(stderr,"Buffer_time requested not set\n");
       
   }
   
   
   temp_rate=rcvd_hw_params->period_time;
   
   fprintf(stderr,"Period Time %u\n", rcvd_hw_params->period_time);
   if((err=snd_pcm_hw_params_set_period_time_near(*handle, params,
                                  &(rcvd_hw_params->period_time), 0)) < 0){
      fprintf(stderr,"alsad_configure_snd_card: ");
      fprintf(stderr,"Could not set period time\n");
      return 0;
   }
   
   fprintf(stderr,"New Period Time %u\n", rcvd_hw_params->period_time);

   if((float)temp_rate * 1.05 < rcvd_hw_params->period_time ||
                          (float)temp_rate * 0.95 > rcvd_hw_params->period_time){
       fprintf(stderr,"alsad_configure_snd_card: ");
       fprintf(stderr,"Period_time requested not set\n");
     
   }
   
   if((err=snd_pcm_hw_params(*handle, params)) < 0){
      fprintf(stderr,"alsad_configure_snd_card: ");
      fprintf(stderr,"Cound not install given hardware parameters\n");
      return 0;
   }

   if((err=snd_pcm_hw_params_get_period_size(params,
                                             &frames_per_period, 0)) < 0){
      fprintf(stderr,"alsad_configure_snd_card: ");
      fprintf(stderr,"Cound not get period size\n");
      return 0;
   }

   rcvd_hw_params->period_size=frames_per_period;


   if((snd_pcm_hw_params_get_buffer_size(params,
                                           &buffer_size_in_frames)) < 0){
      fprintf(stderr,"alsad_configure_snd_card: ");
      fprintf(stderr,"Cound not get buffer size\n");
      return 0;
   }

   rcvd_hw_params->buffer_size=buffer_size_in_frames;

   if(frames_per_period == buffer_size_in_frames){
      fprintf(stderr,"alsad_configure_snd_card: ");
      fprintf(stderr,"Can't use period equal to buffer size\n");
      return 0;
   }

   return 1;
}



int buffer_xrun(snd_pcm_t *handle, long err_val){

   int retval;
   int counter;
   print_pcm_state(handle);
   if(err_val == -EAGAIN){
      fprintf(stderr,"EAGAIN \n"); 
      retval=snd_pcm_wait(handle, ALSAD_PCM_WAIT_TIME);
      if(retval==1){
         fprintf(stderr,"Ready to Read \n"); 
         return 0;
      }
      else{
         fprintf(stderr,"Not ready to read after wait \n"); 
         return -1;
      }
   }
   if(err_val == -EPIPE){
      fprintf(stderr,"EPIPE \n"); 
      if((retval=snd_pcm_prepare(handle))<0){
         fprintf(stderr, "buffer_xrun: ");
         fprintf(stderr, "pcm could not be prepared");
         fprintf(stderr, ": %s\n", snd_strerror(retval));
         return -1;
      }
      retval=snd_pcm_wait(handle, ALSAD_PCM_WAIT_TIME);
      if(retval==1){
         fprintf(stderr,"Ready to Read \n"); 
         return 0;
      }
      else{
         fprintf(stderr,"Not ready to read after wait \n"); 
         return -1;
      }
      return 0; 
   }
   if(err_val == -ESTRPIPE){
      fprintf(stderr,"ESTRPIPE \n"); 
      while((retval=snd_pcm_resume(handle)) == -EAGAIN && counter<
		                               ALSAD_RESUME_COUNT){
         sleep(1);
         counter++;
      }
      if(retval < 0){
         if((retval=snd_pcm_prepare(handle)) < 0){
            fprintf(stderr, "buffer_xrun: ");
            fprintf(stderr, "pcm could not be prepared after resume");
            fprintf(stderr, ": %s\n", snd_strerror(retval));
	         return -1;
         }
      }
   }
   fprintf(stderr,"I think everything is Okay \n"); 
   return 0;
}


void print_pcm_state(snd_pcm_t *handle){

     
   if(snd_pcm_state(handle) == SND_PCM_STATE_XRUN){
      fprintf(stderr, "In state XRUN\n");
   }
   if(snd_pcm_state(handle) == SND_PCM_STATE_SUSPENDED){
      fprintf(stderr, "In state SUSPENDED\n");
   }
   if(snd_pcm_state(handle) == SND_PCM_STATE_OPEN){
      fprintf(stderr, "In state OPEN\n");
   }
   if(snd_pcm_state(handle) == SND_PCM_STATE_PAUSED){
      fprintf(stderr, "In state PAUSED\n");
   }
   if(snd_pcm_state(handle) == SND_PCM_STATE_RUNNING){
      fprintf(stderr, "In state RUNNING\n");
   }
   if(snd_pcm_state(handle) == SND_PCM_STATE_DRAINING){
      fprintf(stderr, "In state DRAINING\n");
   }
   if(snd_pcm_state(handle) == SND_PCM_STATE_PREPARED){
      fprintf(stderr, "In state PREPARED\n");
   }
   if(snd_pcm_state(handle) == SND_PCM_STATE_SETUP){
      fprintf(stderr, "In state SETUP\n");
   }
   
   
   return;
}




void suspend(snd_pcm_t *handle){

   int retval;
   int counter=0;
   fprintf(stderr,"suspend call");

   while((retval=snd_pcm_resume(handle)) == -EAGAIN && counter<
		                               ALSAD_RESUME_COUNT){
      sleep(1);
      counter++;
   }
   if(retval < 0){
      if((retval=snd_pcm_prepare(handle)) < 0){
         fprintf(stderr,"pcm could not be prepared");
         exit(0);
      }
   }
}




int alsad_hw_playback(snd_pcm_t *handle, alsad_stream_props_t *stream_props,
		                      alsad_hw_params_t *configd_hw_params,
                            char *audiobuf){
   int r;
   size_t frames_to_write=configd_hw_params->period_size;
   char *temp_audiobuf=audiobuf;
   int retval;
 
   while(frames_to_write > 0){
      r=snd_pcm_writei(handle, temp_audiobuf, frames_to_write);
      if(r < 0){
         retval=buffer_xrun(handle,r); 
	      if(retval < 0)
            return -1;
          else
            return -1;
      } 
      else{
         frames_to_write -= r;
         temp_audiobuf += r * stream_props->frame_size;
      }
   }

   if(frames_to_write > 0){
      return -1;
   }

   return 0;
}




int  alsad_hw_capture(snd_pcm_t *handle, alsad_stream_props_t *stream_props,
		                     alsad_hw_params_t *configd_hw_params,
                                                         char *audiobuf){
   snd_pcm_sframes_t r;
   snd_pcm_uframes_t frames_to_read=configd_hw_params->period_size;
   char *temp_audiobuf=audiobuf;
   int retval;
   
     //fprintf(stderr,"alsad_hw_capture: frames to read %u\n",(unsigned int)frames_to_read);
         
   while(frames_to_read > 0){
      retval=snd_pcm_wait(handle, ALSAD_PCM_WAIT_TIME);
      r=snd_pcm_readi(handle, temp_audiobuf, frames_to_read);
      if(r >= 0){
         //fprintf(stderr,"alsad_hw_capture: %lu\n",(unsigned long)file_return);
      }
      if(r < 0){
         buffer_xrun(handle,r);
         if(retval < 0)
            return -1;
      } 
      else{
         frames_to_read -= r;
         temp_audiobuf += r * stream_props->frame_size;
      }
   }

         //fprintf(stderr,"alsad_hw_capture: Done with capture\n");

   if(frames_to_read > 0){
      return -1;
   }

   return 0;
}


/*
ssize_t alsad_channel_write(void *channel, void *pipe, char *audiobuf,
	                                	size_t total_to_write){
   ssize_t longretval=0;
   size_t written_bytes=0;
   while(total_to_write > written_bytes && 
         !(alsad_channel_node *)channel->closing &&
         !(alsad_data_pipe *)pipe->closing){
      longretval=write_circ_buff((alsad_channel_node *)channel->audio_buff,
         audiobuf+written_bytes,
         (long)(total_to_write-written_bytes));
      if(longretval<0){
         fprintf(stderr,"Could not write to circ_buff\n");
	 return -1;
      }
      written_bytes+=longretval;
   }
   
   return (ssize_t)written_bytes; 

}
*/

/*
ssize_t alsad_channel_read(void *channel, void *pipe, char *audiobuf,
	                                	size_t total_to_read){
   ssize_t longretval=0;
   size_t read_bytes=0;
   while(total_to_read > read_bytes && 
         !(alsad_channel_node *)channel->closing &&
         !(alsad_data_pipe *)pipe->closing){
      longretval=read_thread_circ_buff((alsad_channel_node *)channel->audio_buff,
	 (alsad_data_pipe *)pipe->thread_byte_id,
         audiobuf+read_bytes,
         (long)(total_to_read-read_bytes));
      if(longretval<0){
         fprintf(stderr,"Could not read from circ_buff\n");
	 return -1;
      }
      read_bytes+=longretval;
   }
   
   return (ssize_t)read_bytes; 

}
*/



ssize_t alsad_file_write(int handle, char *audiobuf, size_t total_to_write){
   ssize_t ret_val;
   char *temp_audiobuf=audiobuf;
   ssize_t total_written=0;
   
   
   while(total_to_write > total_written){
      ret_val=write(handle, temp_audiobuf, total_to_write-total_written);
      if(ret_val == -EPIPE){
         // xrun();
	 fprintf(stderr,"alsad_file_write: EPIPE\n");
         return -1;
      }else if(ret_val == -ESTRPIPE){
         // suspend();
	 fprintf(stderr,"alsad_file_write: ESTRPIPE\n");
         return -1;
      }else if(ret_val<0){
         fprintf(stderr,"alsad_file_write: write error: %s %ld\n",
			                        strerror(errno),(long int)ret_val);
         return -1;
      }
      total_written+=ret_val;
   }
   
   return (ssize_t)total_written;
}



ssize_t alsad_file_read(int handle, char *audiobuf, size_t total_to_read){
   ssize_t ret_val;
   char *temp_audiobuf=audiobuf;
   size_t total_read=0;

   while(total_to_read > total_read){
      ret_val=read(handle, temp_audiobuf, total_to_read-total_read);
      if(ret_val==0){
         break;
      }else if(ret_val<0){
         return -1;
      }
      total_read+=ret_val;
   }

   return total_read;
}


int alsad_strerror(int errnum, char *error_buff, size_t buff_size){

   switch(errnum){
      
      case ALSAD_ERR_MAX_PIPES:
         strncpy(error_buff, 
            "Server cannot add another source or sink. Try another server.",
            buff_size-1);
         break;
         
      case ALSAD_ERR_INSUF_INFO:
         strncpy(error_buff, 
            "Server could not receive the required information for request.",
            buff_size-1);
         break;
         
      case ALSAD_ERR_STREAM_FIND:
         strncpy(error_buff, 
            "Server could not find the requested stream.",
            buff_size-1);      
         break;
         
      case ALSAD_ERR_STREAM_REM:
         strncpy(error_buff, 
            "Server could not remove the requested stream.",
            buff_size-1);            
         break;
         
      case ALSAD_ERR_STREAM_INS:
         strncpy(error_buff, 
            "Could not insert the requested stream. Stream already exists.",
            buff_size-1);
         break;
         
      case ALSAD_ERR_INVLD_FRMT:
         strncpy(error_buff, 
            "Server cannot process the requested sound format.",
            buff_size-1);      
         break;
         
      case ALSAD_ERR_INTERNAL:
         strncpy(error_buff, 
            "Server experienced an internal error. Restart server.",
            buff_size-1);            
         break;

      case ALSAD_ERR_NO_PATH:
         strncpy(error_buff, 
            "Path of data pipe not specified. (CURRENT, REMOTE, HW_IFACE).",
            buff_size-1);            
         break;
      
      case ALSAD_ERR_PIPE_INS:
         strncpy(error_buff, 
            "Could not insert the requested pipe.",
            buff_size-1);            
         break;

      case ALSAD_ERR_PIPE_REM:
         strncpy(error_buff, 
            "Could not remove the requested pipe from the list.",
            buff_size-1);            
         break;
         
      case ALSAD_ERR_PIPE_FIND:
         strncpy(error_buff, 
            "Could not find the requested pipe in the list.",
            buff_size-1);            
         break;

      case ALSAD_ERR_LOCAL_ADDR:
         strncpy(error_buff, 
            "Server could not retrieve its local address.",
            buff_size-1);            
         break;
      
      case ALSAD_ERR_RMTE_HOST:
         strncpy(error_buff, 
            "Server could not connect to remote server.",
            buff_size-1);            
         break;
         
      case ALSAD_ERR_IFACE_CFG:
         strncpy(error_buff, 
            "Server could not configure the supplied hw_iface.",
            buff_size-1);            
         break;
         
      case ALSAD_ERR_RECV_CTRL:
         strncpy(error_buff, 
            "Could not receive control message.  Timeout occurred.",
            buff_size-1);            
         break;

      case ALSAD_ERR_INVLD_CTRL:
         strncpy(error_buff, 
            "Server received an invalid control message.",
            buff_size-1);            
         break;

      case ALSAD_ERR_RECV_AUDIOHD:
         strncpy(error_buff, 
            "Could not receive audio header. Broken protocol.",
            buff_size-1);            
         break;

      case ALSAD_ERR_MAX_TRANS:
         strncpy(error_buff, 
            "Client exceeded the requested or maximum transmission size.",
            buff_size-1);            
         break;
      
      case ALSAD_ERR_FRAME_DIV:
         strncpy(error_buff, 
            "Received audio data is not divisible by the frame size",
            buff_size-1);            
         break;
      
      case ALSAD_ERR_RECV_AUDIO:
         strncpy(error_buff, 
            "Could not receive audio data from client. Broken protocol.",
            buff_size-1);            
         break;

      case ALSAD_ERR_PIPE_CLOSE:
         strncpy(error_buff, 
            "Pipe has been closed by request.",
            buff_size-1);            
         break;

      case ALSAD_ERR_STREAM_CLOSE:
         strncpy(error_buff, 
            "Stream closed.",
            buff_size-1);            
         break;

      case ALSAD_ERR_SHUTDOWN:
         strncpy(error_buff, 
            "Shutdown signal sent to alsad.",
            buff_size-1);            
         break;
      
      case ALSAD_ERR_SRC_PERM:
         strncpy(error_buff, 
            "Client did not set the source permissions.",
            buff_size-1);            
         break;
      
      
      default:
         return -1;
         
   }
   return 0;
}

/** @}*/
