/*! \file alsad_net_lib.c
 *  \brief This is the implementation of the Communications Library
 *  \author Daniel R. Warren
 *  \version 1.0
 *  \date    November 2004
 *  \ingroup net_lib
*/


 /** \defgroup net_lib The Communications Library
 *   \ingroup application_components
 *   \brief Library that provides a mechanism for communicating between
 *          the client and alsad.
 * 
   
 */
 
/** \addtogroup net_lib */
/** @{*/



#include "alsad_net_lib.h"



void alsad_initialize_hw_params(alsad_hw_params_t *init_hw_params){

   init_hw_params->stream=0;
   init_hw_params->open_mode=0;
   init_hw_params->buffer_time=0;
   init_hw_params->period_time=0;
   init_hw_params->buffer_size=0;
   init_hw_params->period_size=0;
   memset(init_hw_params->pcm_name,0,ALSAD_MAX_DEV_LENGTH);

}	



void alsad_initialize_sw_params(alsad_sw_params_t *init_sw_params){ 

   init_sw_params->foo=0;
   init_sw_params->bar=0;

}


void alsad_initialize_connect_addr(alsad_connect_addr_t *init_connect_addr){ 

   init_connect_addr->port=0;
   memset(init_connect_addr->hostname,'\0',ALSAD_MAX_HOST_LEN);

}


void alsad_initialize_stream_props(alsad_stream_props_t *init_stream_props){ 

   init_stream_props->identifier=0;
   init_stream_props->format=0;
   init_stream_props->channels=0;
   init_stream_props->rate=0;
   init_stream_props->frame_size=0;
   init_stream_props->circ_buff_size=0;
   init_stream_props->src_permission=0;
   memset(init_stream_props->text_desc,0,ALSAD_MAX_TEXT_DESC); 

}


void alsad_initialize_data_pipe(alsad_data_pipe_t *init_data_pipe){ 

   init_data_pipe->identifier=0;
   init_data_pipe->connect_addr=NULL;
   init_data_pipe->hw_params=NULL;
   init_data_pipe->sw_params=NULL;

}
	
int alsad_send_hw_params(int sock, alsad_hw_params_t *hw_params){
   
   alsad_control_t *temp_control;
   alsad_control_t hard_control;
   alsad_hw_params_t *temp_hw_params;
   char *temp_buff;
   int ret_val;

   hard_control.ctrl=ALSAD_RECV_HW_PARAMS; 
   temp_buff = (char *)malloc(sizeof(alsad_control_t)+sizeof(alsad_hw_params_t));
   if (temp_buff == NULL) return 0;
   memset(temp_buff, 0, sizeof(alsad_control_t)+sizeof(alsad_hw_params_t));
   temp_control = (alsad_control_t *)temp_buff;
   temp_hw_params = (alsad_hw_params_t *)(temp_buff + sizeof(alsad_control_t));
   
   temp_control->ctrl = htonl(hard_control.ctrl);
   temp_control->code = htonl(hard_control.code);
   memcpy(temp_hw_params, hw_params, sizeof(alsad_hw_params_t));
   
   temp_hw_params->stream = htonl(hw_params->stream);
   temp_hw_params->open_mode = htonl(hw_params->open_mode);
   temp_hw_params->buffer_time = htonl(hw_params->buffer_time);
   temp_hw_params->period_time = htonl(hw_params->period_time);
   temp_hw_params->buffer_size = htonl(hw_params->buffer_size);
   temp_hw_params->period_size = htonl(hw_params->period_size);
   
   ret_val = safe_sock_send(sock,temp_buff,sizeof(alsad_control_t)+
		                                sizeof(alsad_hw_params_t),
                                       ALSAD_GENERIC_SOCK_TIMEOUT);
   
   free(temp_buff);
   
   return ret_val;
}


int alsad_recv_hw_params(int sock, alsad_hw_params_t *hw_params){

   int ret_val;

   ret_val = safe_sock_recv(sock, hw_params, sizeof(alsad_hw_params_t),
                                           ALSAD_GENERIC_SOCK_TIMEOUT);
   if(ret_val <= 0)
      return ret_val;

    
   hw_params->stream = ntohl(hw_params->stream);
   hw_params->open_mode = ntohl(hw_params->open_mode);
   hw_params->buffer_time = ntohl(hw_params->buffer_time);
   hw_params->period_time = ntohl(hw_params->period_time);
   hw_params->buffer_size = ntohl(hw_params->buffer_size);
   hw_params->period_size = ntohl(hw_params->period_size);
   
   
   return ret_val;  
}


int alsad_send_hw_params_w_ctrl(int sock, alsad_hw_params_t *hw_params,
		                       alsad_control_t *send_control){
   
   alsad_control_t *temp_control;
   alsad_hw_params_t *temp_hw_params;
   char *temp_buff;
   int ret_val;

   temp_buff = (char *)malloc(sizeof(alsad_control_t)+sizeof(alsad_hw_params_t));
   if (temp_buff == NULL) return 0;
   memset(temp_buff, 0, sizeof(alsad_control_t)+sizeof(alsad_hw_params_t));
   temp_control = (alsad_control_t *)temp_buff;
   temp_hw_params = (alsad_hw_params_t *)(temp_buff + sizeof(alsad_control_t));
   
   temp_control->ctrl = htonl(send_control->ctrl);
   temp_control->code = htonl(send_control->code);
   memcpy(temp_hw_params, hw_params, sizeof(alsad_hw_params_t));
   
   temp_hw_params->stream = htonl(hw_params->stream);
   temp_hw_params->open_mode = htonl(hw_params->open_mode);
   temp_hw_params->buffer_time = htonl(hw_params->buffer_time);
   temp_hw_params->period_time = htonl(hw_params->period_time);
   temp_hw_params->buffer_size = htonl(hw_params->buffer_size);
   temp_hw_params->period_size = htonl(hw_params->period_size);
   
   ret_val = safe_sock_send(sock,temp_buff,sizeof(alsad_control_t)+
		                                sizeof(alsad_hw_params_t),
                                       ALSAD_GENERIC_SOCK_TIMEOUT);
   
   free(temp_buff);
   
   return ret_val;
}



int alsad_send_sw_params(int sock, alsad_sw_params_t *sw_params){
   alsad_control_t send_control;
   send_control.ctrl=ALSAD_RECV_SW_PARAMS;
   send_control.code=0;
   return alsad_send_control(sock,&send_control);
   
}



int alsad_recv_sw_params(int sock, alsad_sw_params_t *sw_params){
   return 1;
}


int alsad_send_sw_params_w_ctrl(int sock, alsad_sw_params_t *sw_params,
		                      alsad_control_t *send_control){
   return alsad_send_control(sock,send_control);
}



int alsad_send_audio_hdr(int sock, alsad_audio_hdr_t *audio_hdr,
		                      char *audiobuf){
   
   alsad_control_t audio_control;
   alsad_control_t *temp_control;
   alsad_audio_hdr_t *temp_audio_hdr;
   char *temp_buff;
   char *temp_audiobuf;
   unsigned int temp_buff_size;
   int ret_val;
   
   audio_control.ctrl=ALSAD_RECV_AUDIO_BUF;
   temp_buff_size = sizeof(alsad_control_t)+sizeof(alsad_audio_hdr_t)+
	                                          audio_hdr->payload_size;
   
   temp_buff = (char *)malloc(temp_buff_size);
   if (temp_buff == NULL) return 0;
   memset(temp_buff, 0, sizeof(alsad_control_t)+sizeof(alsad_audio_hdr_t));
   temp_control = (alsad_control_t *)temp_buff;
   temp_audio_hdr = (alsad_audio_hdr_t *)(temp_buff + sizeof(alsad_control_t));
   temp_audiobuf = (char *)(temp_buff+sizeof(alsad_control_t)+
		                                 sizeof(alsad_audio_hdr_t));
   
   memcpy(temp_audiobuf, audiobuf, audio_hdr->payload_size);
   
   temp_control->ctrl = htonl(audio_control.ctrl); 
   temp_control->code = htonl(audio_control.code); 

   temp_audio_hdr->payload_size = htonl(audio_hdr->payload_size);
   
   ret_val = safe_sock_send(sock, temp_buff, temp_buff_size,
                                               ALSAD_GENERIC_SOCK_TIMEOUT);
   
   free(temp_buff);
   
   return ret_val;
}


int alsad_recv_audio_hdr(int sock, alsad_audio_hdr_t *audio_hdr){

   int ret_val;

   ret_val = safe_sock_recv(sock,audio_hdr,sizeof(alsad_audio_hdr_t),
                                              ALSAD_GENERIC_SOCK_TIMEOUT);
   if(ret_val <= 0)
      return ret_val;
   
   audio_hdr->payload_size = ntohl(audio_hdr->payload_size);
   
   return ret_val;  

}



int alsad_send_stream_props(int sock, alsad_stream_props_t *stream_props){

   alsad_control_t *temp_control;
   alsad_control_t audio_control;
   alsad_stream_props_t *temp_stream_props;
   char *temp_buff;
   int ret_val;

   audio_control.ctrl=ALSAD_RECV_STREAM_PROPS;
   temp_buff = (char *)malloc(sizeof(alsad_control_t)+
		                       sizeof(alsad_stream_props_t));
   if (temp_buff == NULL) return 0;
   memset(temp_buff, 0, sizeof(alsad_control_t)+sizeof(alsad_stream_props_t));
   temp_control = (alsad_control_t *)temp_buff;
   temp_stream_props = (alsad_stream_props_t*)(temp_buff + sizeof(alsad_control_t));
   
   temp_control->ctrl = htonl(audio_control.ctrl);
   temp_control->code = htonl(audio_control.code);
   memcpy(temp_stream_props, stream_props, sizeof(alsad_stream_props_t));
   
   temp_stream_props->identifier = htonl(stream_props->identifier);
   temp_stream_props->format = htonl(stream_props->format);
   temp_stream_props->channels  = htonl(stream_props->channels);
   temp_stream_props->rate = htonl(stream_props->rate);
   temp_stream_props->frame_size = htonl(stream_props->frame_size);
   temp_stream_props->circ_buff_size = htonl(stream_props->circ_buff_size);
   temp_stream_props->src_permission = htonl(stream_props->src_permission);
   
   ret_val = safe_sock_send(sock,temp_buff,sizeof(alsad_control_t)+
		                              sizeof(alsad_stream_props_t),
                                     ALSAD_GENERIC_SOCK_TIMEOUT);
   
   free(temp_buff);
   
   return ret_val;
   

}



int alsad_recv_stream_props(int sock, alsad_stream_props_t *stream_props){

   int ret_val;

   ret_val = safe_sock_recv(sock,stream_props,sizeof(alsad_stream_props_t),
                                                 ALSAD_GENERIC_SOCK_TIMEOUT);
   if(ret_val <= 0)
      return ret_val;
   
   stream_props->identifier = ntohl(stream_props->identifier);
   stream_props->format = ntohl(stream_props->format);
   stream_props->channels  = ntohl(stream_props->channels);
   stream_props->rate = ntohl(stream_props->rate);
   stream_props->frame_size = ntohl(stream_props->frame_size);
   stream_props->circ_buff_size = ntohl(stream_props->circ_buff_size);
   stream_props->src_permission = ntohl(stream_props->src_permission);

   
   return ret_val;  
}


int alsad_send_conn_addr(int sock, alsad_connect_addr_t *conn_addr){

   alsad_control_t *temp_control;
   alsad_control_t connect_control;
   alsad_connect_addr_t *temp_conn_addr;
   char *temp_buff;
   int ret_val;

   connect_control.ctrl=ALSAD_RECV_CONN_ADDR;
   temp_buff = (char *)malloc(sizeof(alsad_control_t)+
		                            sizeof(alsad_connect_addr_t));
   if (temp_buff == NULL) return 0;
   memset(temp_buff, 0, sizeof(alsad_control_t)+sizeof(alsad_connect_addr_t));
   temp_control = (alsad_control_t *)temp_buff;
   temp_conn_addr = (alsad_connect_addr_t*)(temp_buff + sizeof(alsad_control_t));
   
   temp_control->ctrl = htonl(connect_control.ctrl);
   temp_control->code = htonl(connect_control.code);
   memcpy(temp_conn_addr, conn_addr, sizeof(alsad_connect_addr_t));
   
   temp_conn_addr->port = htonl(conn_addr->port);
   
   ret_val = safe_sock_send(sock,temp_buff,sizeof(alsad_control_t)+
		                              sizeof(alsad_connect_addr_t),
                                       ALSAD_GENERIC_SOCK_TIMEOUT);
   
   free(temp_buff);
   
   return ret_val;
   

}




int alsad_recv_conn_addr(int sock, alsad_connect_addr_t *conn_addr){
   int ret_val;

   ret_val = safe_sock_recv(sock,conn_addr,sizeof(alsad_connect_addr_t),
                                           ALSAD_GENERIC_SOCK_TIMEOUT);
   if(ret_val <= 0)
      return ret_val;
   
   conn_addr->port = ntohl(conn_addr->port );

   
   return ret_val;  
}




int alsad_send_conn_addr_w_ctrl(int sock, alsad_connect_addr_t *conn_addr,
		                       alsad_control_t *send_control){

   alsad_control_t *temp_control;
   alsad_connect_addr_t *temp_conn_addr;
   char *temp_buff;
   int ret_val;

   temp_buff = (char *)malloc(sizeof(alsad_control_t)+
		                            sizeof(alsad_connect_addr_t));
   if (temp_buff == NULL) return 0;
   memset(temp_buff, 0, sizeof(alsad_control_t)+sizeof(alsad_connect_addr_t));
   temp_control = (alsad_control_t *)temp_buff;
   temp_conn_addr = (alsad_connect_addr_t*)(temp_buff + sizeof(alsad_control_t));
   
   temp_control->ctrl = htonl(send_control->ctrl);
   temp_control->code = htonl(send_control->code);
   memcpy(temp_conn_addr, conn_addr, sizeof(alsad_connect_addr_t));
   
   temp_conn_addr->port = htonl(conn_addr->port);
   
   ret_val = safe_sock_send(sock,temp_buff,sizeof(alsad_control_t)+
		                              sizeof(alsad_connect_addr_t),
                                       ALSAD_GENERIC_SOCK_TIMEOUT);
   
   free(temp_buff);
   
   return ret_val;
   

}



int alsad_send_control(int sock, alsad_control_t *control){
   
   alsad_control_t *temp_control;
   int ret_val;

   temp_control = (alsad_control_t*)malloc(sizeof(alsad_control_t));
   if (temp_control == NULL) return 0;
   memset(temp_control, 0, sizeof(alsad_control_t));
   
   temp_control->ctrl = htonl(control->ctrl);
   temp_control->code = htonl(control->code);
   
   ret_val = safe_sock_send(sock,temp_control,sizeof(alsad_control_t),
                                              ALSAD_GENERIC_SOCK_TIMEOUT);
   
   free(temp_control);
   
   return ret_val;
}


int alsad_recv_control(int sock, alsad_control_t *control){

   int ret_val;

   ret_val = safe_sock_recv(sock,control,sizeof(alsad_control_t),
                                          ALSAD_GENERIC_SOCK_TIMEOUT);
   if(ret_val <= 0)
      return ret_val;
   control->ctrl = ntohl(control->ctrl);
   control->code = ntohl(control->code);
   
   return ret_val;  


}



int alsad_send_data_pipe(int sock, alsad_data_pipe_t *data_pipe){

   alsad_control_t *temp_control;
   alsad_control_t data_pipe_control;
   alsad_data_pipe_t *temp_data_pipe;
   char *temp_buff;
   int ret_val;

   data_pipe_control.ctrl=ALSAD_RECV_DATA_PIPE;
   temp_buff = (char *)malloc(sizeof(alsad_control_t)+
		                            sizeof(alsad_data_pipe_t));
   if (temp_buff == NULL) return 0;
   memset(temp_buff, 0, sizeof(alsad_control_t)+sizeof(alsad_data_pipe_t));
   temp_control = (alsad_control_t *)temp_buff;
   temp_data_pipe = (alsad_data_pipe_t*)(temp_buff + sizeof(alsad_control_t));
   
   temp_control->ctrl = htonl(data_pipe_control.ctrl);
   temp_control->code = htonl(data_pipe_control.code);
   
   memcpy(temp_data_pipe, data_pipe, sizeof(alsad_data_pipe_t));
   
   temp_data_pipe->identifier = htonl(data_pipe->identifier);
   
   ret_val = safe_sock_send(sock,temp_buff,sizeof(alsad_control_t)+
		                              sizeof(unsigned int),
                                       ALSAD_GENERIC_SOCK_TIMEOUT);
   
   free(temp_buff);
   
   return ret_val;
   

}



/**TODO Send and Recv data pipe should send all of the data
 * not just the identifier.
 */
 
int alsad_recv_data_pipe(int sock, alsad_data_pipe_t *data_pipe){
   int ret_val;

   ret_val = safe_sock_recv(sock,data_pipe,sizeof(unsigned int),
                                            ALSAD_GENERIC_SOCK_TIMEOUT);
   if(ret_val <= 0)
      return ret_val;
   
   data_pipe->identifier = ntohl(data_pipe->identifier);

   return ret_val;  
}



int alsad_send_data_pipe_w_ctrl(int sock, alsad_data_pipe_t *data_pipe,
		                         alsad_control_t *send_control){

   alsad_control_t *temp_control;
   alsad_data_pipe_t *temp_data_pipe;
   char *temp_buff;
   int ret_val;

   temp_buff = (char *)malloc(sizeof(alsad_control_t)+
		                            sizeof(alsad_data_pipe_t));
   if (temp_buff == NULL) return 0;
   memset(temp_buff, 0, sizeof(alsad_control_t)+sizeof(alsad_data_pipe_t));
   temp_control = (alsad_control_t *)temp_buff;
   temp_data_pipe = (alsad_data_pipe_t*)(temp_buff + sizeof(alsad_control_t));
   
   temp_control->ctrl = htonl(send_control->ctrl);
   temp_control->code = htonl(send_control->code);
   
   memcpy(temp_data_pipe, data_pipe, sizeof(alsad_data_pipe_t));
   
   temp_data_pipe->identifier = htonl(data_pipe->identifier);
   
   ret_val = safe_sock_send(sock,temp_buff,sizeof(alsad_control_t)+
		                              sizeof(unsigned int),
                                      ALSAD_GENERIC_SOCK_TIMEOUT);
   
   free(temp_buff);
   
   return ret_val;
   

}



int alsad_request_structs(int sock, alsad_net_structs_t *net_structs, 
		                               alsad_control_t *control){
   alsad_control_t rcvd_control;
   alsad_control_t send_control;
   alsad_control_t rcvd_all_control;
   int retval;

   rcvd_all_control.ctrl=0;

   if((retval=alsad_send_control(sock, control)) <= 0){;
      fprintf(stderr,
              "alsad_request_structs: Could not send control message.\n");
      return -1;
   }

   if((retval=alsad_recv_control(sock, &rcvd_control)) <=0){
      fprintf(stderr,"alsad_request_structs: Failed to recv control.\n");
      return -1;
   }
   

   while((rcvd_control.ctrl & ALSAD_MSG_RQST_OK) != ALSAD_MSG_RQST_OK && 
         (rcvd_control.ctrl & ALSAD_MSG_RQST_ERR) != ALSAD_MSG_RQST_ERR){ 
      
	  /* 
	   // Did I get something that I didn't ask for?
      if((rcvd_control.ctrl & control->ctrl) == 0){
         fprintf(stdout, "I got something that I didn't ask for\n"); 
	 control->ctrl=(control->ctrl | ALSAD_MSG_RQST_ERR);
         if((retval=alsad_send_control(sock, control)) <= 0){;
            fprintf(stderr,"alsad_request_structs: ");
            fprintf(stderr,"Could not send control message.\n");
         }
         return -1;
      }

*/

      if((rcvd_control.ctrl & ALSAD_RECV_HW_PARAMS) == ALSAD_RECV_HW_PARAMS){

	 if(net_structs->hw_params==NULL){
            fprintf(stderr,"alsad_request_structs: ");
            fprintf(stderr,"hw_params buffer not initialized.\n");
            return -1;
         }
         if((retval=alsad_recv_hw_params(sock, net_structs->hw_params))<=0){
            fprintf(stderr,"alsad_request_structs: ");
            fprintf(stderr,"Unable to recv hardware parameters from client\n");
            return -1;
         }
         rcvd_all_control.ctrl=(rcvd_all_control.ctrl |
			                        ALSAD_SEND_HW_PARAMS); 

      }
     

      else if((rcvd_control.ctrl & ALSAD_RECV_SW_PARAMS) == ALSAD_RECV_SW_PARAMS){

	 if(net_structs->sw_params==NULL){
            fprintf(stderr,"alsad_request_structs: ");
            fprintf(stderr,"sw_params buffer not initialized.\n");
            return -1;
         }
         if((retval=alsad_recv_sw_params(sock, net_structs->sw_params))<=0){
            fprintf(stderr,"alsad_request_structs: ");
            fprintf(stderr,"Unable to recv software parameters from client\n");
            return -1;
         }
         rcvd_all_control.ctrl=(rcvd_all_control.ctrl |
			                        ALSAD_SEND_SW_PARAMS); 
      }
        
      
      else if((rcvd_control.ctrl & ALSAD_RECV_CONN_ADDR)
		                           == ALSAD_RECV_CONN_ADDR){

	 if(net_structs->connect_addr==NULL){
            fprintf(stderr,"alsad_request_structs: ");
            fprintf(stderr,"connect_addr buffer not initialized.\n");
            return -1;
         }
         if((retval=alsad_recv_conn_addr(sock, 
				      net_structs->connect_addr))<=0){
            fprintf(stderr,"alsad_request_structs: ");
            fprintf(stderr,"Unable to recv connection address from client\n");
            return -1;
         }
         rcvd_all_control.ctrl=(rcvd_all_control.ctrl |
			                        ALSAD_SEND_CONN_ADDR); 
      }
      
      
      else if((rcvd_control.ctrl & ALSAD_RECV_LOCAL_STREAM)
                                            == ALSAD_RECV_LOCAL_STREAM){
      
         if((retval=alsad_recv_control(sock, &rcvd_control)) <=0){
            fprintf(stderr,"alsad_request_structs: Failed to recv control.\n");
            return -1;
         }
         if((rcvd_control.ctrl & ALSAD_RECV_STREAM_PROPS)
                                            == ALSAD_RECV_STREAM_PROPS){
            if(net_structs->local_stream==NULL){
               fprintf(stderr,"alsad_request_structs: ");
               fprintf(stderr,"local_stream buffer not initialized.\n");
               return -1;
            }
            if((retval=alsad_recv_stream_props(sock,
				      net_structs->local_stream))<=0){
               fprintf(stderr,"alsad_request_structs: ");
               fprintf(stderr,"Unable to recv local stream from client\n");
               return -1;
            }
            rcvd_all_control.ctrl=(rcvd_all_control.ctrl |
			                        ALSAD_SEND_LOCAL_STREAM); 

         }
         else{
           fprintf(stderr,"alsad_request_structs: ");
           fprintf(stderr,"protocol violation receiving LOCAL_STREAM\n");
           rcvd_control.ctrl=ALSAD_MSG_RQST_ERR;
           continue;
         }      
      }



      else if((rcvd_control.ctrl & ALSAD_RECV_RMTE_STREAM)
                                            == ALSAD_RECV_RMTE_STREAM){
         if((retval=alsad_recv_control(sock, &rcvd_control)) <=0){
            fprintf(stderr,"alsad_request_structs: Failed to recv control.\n");
            return -1;
         }
         if((rcvd_control.ctrl & ALSAD_RECV_STREAM_PROPS)
                                            == ALSAD_RECV_STREAM_PROPS){
            if(net_structs->remote_stream==NULL){
               fprintf(stderr,"alsad_request_structs: ");
               fprintf(stderr,"remote_stream buffer not initialized.\n");
               return -1;
            }
            if((retval=alsad_recv_stream_props(sock,
				      net_structs->remote_stream))<=0){
               fprintf(stderr,"alsad_request_structs: ");
               fprintf(stderr,"Unable to recv remote channel from client\n");
               return -1;
            }
            rcvd_all_control.ctrl=(rcvd_all_control.ctrl |
			                        ALSAD_SEND_RMTE_STREAM); 

         }      
         else{
           fprintf(stderr,"alsad_request_structs: ");
           fprintf(stderr,"protocol violation receiving RMTE_STREAM\n");
           rcvd_control.ctrl=ALSAD_MSG_RQST_ERR;
           continue;
         }      
      }



      else if((rcvd_control.ctrl & ALSAD_RECV_LOCAL_PIPE)
                                            == ALSAD_RECV_LOCAL_PIPE){
         if((retval=alsad_recv_control(sock, &rcvd_control)) <=0){
            fprintf(stderr,"alsad_request_structs: Failed to recv control.\n");
            return -1;
         }
         if((rcvd_control.ctrl & ALSAD_RECV_DATA_PIPE)
		                           == ALSAD_RECV_DATA_PIPE){
	    if(net_structs->local_pipe==NULL){
               fprintf(stderr,"alsad_request_structs: ");
               fprintf(stderr,"local_pipe buffer not initialized.\n");
               return -1;
            }
            if((retval=alsad_recv_data_pipe(sock, 
	 			      net_structs->local_pipe))<=0){
               fprintf(stderr,"alsad_request_structs: ");
               fprintf(stderr,"Unable to recv local pipe from client\n");
               return -1;
            }
            rcvd_all_control.ctrl=(rcvd_all_control.ctrl |
			                        ALSAD_SEND_LOCAL_PIPE); 
         }
         else{
           fprintf(stderr,"alsad_request_structs: ");
           fprintf(stderr,"protocol violation receiving LOCAL_PIPE\n");
           rcvd_control.ctrl=ALSAD_MSG_RQST_ERR;
           continue;
         }      
      
      }



      else if((rcvd_control.ctrl & ALSAD_RECV_RMTE_PIPE)
                                            == ALSAD_RECV_RMTE_PIPE){
         if((retval=alsad_recv_control(sock, &rcvd_control)) <=0){
            fprintf(stderr,"alsad_request_structs: Failed to recv control.\n");
            return -1;
         }
         if((rcvd_control.ctrl & ALSAD_RECV_DATA_PIPE)
		                           == ALSAD_RECV_DATA_PIPE){
	    if(net_structs->remote_pipe==NULL){
               fprintf(stderr,"alsad_request_structs: ");
               fprintf(stderr,"remote_pipe buffer not initialized.\n");
               return -1;
            }
            if((retval=alsad_recv_data_pipe(sock, 
	 			      net_structs->remote_pipe))<=0){
               fprintf(stderr,"alsad_request_structs: ");
               fprintf(stderr,"Unable to recv remote pipe from client\n");
               return -1;
            }
            rcvd_all_control.ctrl=(rcvd_all_control.ctrl |
			                        ALSAD_SEND_RMTE_PIPE); 
         }
         else{
           fprintf(stderr,"alsad_request_structs: ");
           fprintf(stderr,"protocol violation receiving RMTE_PIPE\n");
           rcvd_control.ctrl=ALSAD_MSG_RQST_ERR;
           continue;
         }      
      }
    
    
      if((retval=alsad_recv_control(sock, &rcvd_control)) <=0){
         fprintf(stderr,"alsad_request_structs: Failed to recv control.\n");
         return -1;
      }
   
   }
   
   if((rcvd_control.ctrl & ALSAD_MSG_RQST_ERR) == ALSAD_MSG_RQST_ERR){ 
      fprintf(stderr,"alsad_request_structs: Received error message exiting.\n");
      return -1;  
   }
         
   if((rcvd_all_control.ctrl & control->ctrl) != control->ctrl){ 
      fprintf(stderr,"alsad_request_structs: Did not receive all structs.\n");
      return -1;  
   }

   send_control.ctrl=ALSAD_MSG_RQST_OK; 
   if((retval=alsad_send_control(sock, &send_control)) <= 0){;
      fprintf(stderr,
              "alsad_reply_structs: Could not send control message.\n");
      return -1;  
   }
   return 0; 
}





int alsad_reply_structs(int sock, alsad_net_structs_t *net_structs){

   alsad_control_t rcvd_control;
   alsad_control_t send_control;
   int error_occured;
   int retval;


   error_occured=0;
   
   if((retval=alsad_recv_control(sock, &rcvd_control)) <= 0){;
      fprintf(stderr,
              "alsad_reply_structs: Could not receive control message.\n");
      return -1;
   }

   while((rcvd_control.ctrl & ALSAD_MSG_RQST_OK) != ALSAD_MSG_RQST_OK && 
         (rcvd_control.ctrl & ALSAD_MSG_RQST_ERR) != ALSAD_MSG_RQST_ERR){ 
         
      if((rcvd_control.ctrl & ALSAD_SEND_HW_PARAMS) == ALSAD_SEND_HW_PARAMS){

	 if(net_structs->hw_params==NULL){
            fprintf(stderr,"alsad_reply_structs: ");
            fprintf(stderr,"hw_params buffer not initialized.\n");
            send_control.ctrl=ALSAD_MSG_RQST_ERR; 
            if((retval=alsad_send_control(sock, &send_control)) <= 0){;
               fprintf(stderr,
                    "alsad_reply_structs: Could not send control message.\n");
               return -1;
            }
            return -1;
         }
         if((retval=alsad_send_hw_params(sock, net_structs->hw_params))<=0){
            fprintf(stderr,"alsad_reply_structs: ");
            fprintf(stderr,"Unable to send hardware parameters to client\n");
            return -1;
         }
      }
     

      if((rcvd_control.ctrl & ALSAD_SEND_SW_PARAMS) == ALSAD_SEND_SW_PARAMS){

	 if(net_structs->sw_params==NULL){
            fprintf(stderr,"alsad_reply_structs: ");
            fprintf(stderr,"sw_params buffer not initialized.\n");
            send_control.ctrl=ALSAD_MSG_RQST_ERR; 
            if((retval=alsad_send_control(sock, &send_control)) <= 0){;
               fprintf(stderr,
                    "alsad_reply_structs: Could not send control message.\n");
               return -1;
            }
            return -1;
         }
         if((retval=alsad_send_sw_params(sock, net_structs->sw_params))<=0){
            fprintf(stderr,"alsad_reply_structs: ");
            fprintf(stderr,"Unable to send software parameters to client\n");
            return -1;
         }
      }
  

      if((rcvd_control.ctrl & ALSAD_SEND_LOCAL_STREAM)
		                           == ALSAD_SEND_LOCAL_STREAM){

	 if(net_structs->local_stream==NULL){
            fprintf(stderr,"alsad_reply_structs: ");
            fprintf(stderr,"local_stream buffer not initialized.\n");
            send_control.ctrl=ALSAD_MSG_RQST_ERR; 
            if((retval=alsad_send_control(sock, &send_control)) <= 0){;
               fprintf(stderr,
                    "alsad_reply_structs: Could not send control message.\n");
               return -1;
            }
            return -1;
         }

         send_control.ctrl=ALSAD_RECV_LOCAL_STREAM;
         if((retval=alsad_send_control(sock, &send_control)) <= 0){;
            fprintf(stderr,
                 "alsad_reply_structs: Could not send control message.\n");
            return -1;
         }
         
         if((retval=alsad_send_stream_props(sock,
				      net_structs->local_stream))<=0){
            fprintf(stderr,"alsad_reply_structs: ");
            fprintf(stderr,"Unable to send local channel to client\n");
            return -1;
         }
      }
      
      
      if((rcvd_control.ctrl & ALSAD_SEND_RMTE_STREAM)
		                           == ALSAD_SEND_RMTE_STREAM){

	      if(net_structs->remote_stream==NULL){
            fprintf(stderr,"alsad_reply_structs: ");
            fprintf(stderr,"remote_stream buffer not initialized.\n");
            send_control.ctrl=ALSAD_MSG_RQST_ERR; 
            if((retval=alsad_send_control(sock, &send_control)) <= 0){;
               fprintf(stderr,
                    "alsad_reply_structs: Could not send control message.\n");
               return -1;
            }
            return -1;
         }

         send_control.ctrl=ALSAD_RECV_RMTE_STREAM;
         if((retval=alsad_send_control(sock, &send_control)) <= 0){;
            fprintf(stderr,
                 "alsad_reply_structs: Could not send control message.\n");
            return -1;
         }
         
         if((retval=alsad_send_stream_props(sock,
				      net_structs->remote_stream))<=0){
            fprintf(stderr,"alsad_reply_structs: ");
            fprintf(stderr,"Unable to send remote channel to client\n");
            return -1;
         }
      }
      
      if((rcvd_control.ctrl & ALSAD_SEND_CONN_ADDR)
		                           == ALSAD_SEND_CONN_ADDR){

	 if(net_structs->connect_addr==NULL){
            fprintf(stderr,"alsad_reply_structs: ");
            fprintf(stderr,"connect_addr buffer not initialized.\n");
            send_control.ctrl=ALSAD_MSG_RQST_ERR; 
            if((retval=alsad_send_control(sock, &send_control)) <= 0){;
               fprintf(stderr,
                    "alsad_reply_structs: Could not send control message.\n");
               return -1;
            }
            return -1;
         }
         if((retval=alsad_send_conn_addr(sock, 
				      net_structs->connect_addr))<=0){
            fprintf(stderr,"alsad_reply_structs: ");
            fprintf(stderr,"Unable to send connection address to client\n");
            return -1;
         }
      }
      
      
      if((rcvd_control.ctrl & ALSAD_SEND_LOCAL_PIPE)
		                           == ALSAD_SEND_LOCAL_PIPE){

	 if(net_structs->local_pipe==NULL){
            fprintf(stderr,"alsad_reply_structs: ");
            fprintf(stderr,"local_pipe buffer not initialized.\n");
            send_control.ctrl=ALSAD_MSG_RQST_ERR; 
            if((retval=alsad_send_control(sock, &send_control)) <= 0){;
               fprintf(stderr,
                    "alsad_reply_structs: Could not send control message.\n");
               return -1;
            }
            return -1;
         }

         send_control.ctrl=ALSAD_RECV_LOCAL_PIPE;
         if((retval=alsad_send_control(sock, &send_control)) <= 0){;
            fprintf(stderr,
                 "alsad_reply_structs: Could not send control message.\n");
            return -1;
         }
         
         if((retval=alsad_send_data_pipe(sock,
				      net_structs->local_pipe))<=0){
            fprintf(stderr,"alsad_reply_structs: ");
            fprintf(stderr,"Unable to send local pipe to client\n");
            return -1;
         }
      }
      
      
      if((rcvd_control.ctrl & ALSAD_SEND_RMTE_PIPE)
		                           == ALSAD_SEND_RMTE_PIPE){

	 if(net_structs->remote_stream==NULL){
            fprintf(stderr,"alsad_reply_structs: ");
            fprintf(stderr,"remote_pipe buffer not initialized.\n");
            send_control.ctrl=ALSAD_MSG_RQST_ERR; 
            if((retval=alsad_send_control(sock, &send_control)) <= 0){;
               fprintf(stderr,
                    "alsad_reply_structs: Could not send control message.\n");
               return -1;
            }
            return -1;
         }

         send_control.ctrl=ALSAD_RECV_RMTE_PIPE;
         if((retval=alsad_send_control(sock, &send_control)) <= 0){;
            fprintf(stderr,
                 "alsad_reply_structs: Could not send control message.\n");
            return -1;
         }
         
         if((retval=alsad_send_data_pipe(sock,
				      net_structs->remote_pipe))<=0){
            fprintf(stderr,"alsad_reply_structs: ");
            fprintf(stderr,"Unable to send remote pipe to client\n");
            return -1;
         }
      }
      

      send_control.ctrl=ALSAD_MSG_RQST_OK; 
      if((retval=alsad_send_control(sock, &send_control)) <= 0){;
         fprintf(stderr,
              "alsad_reply_structs: Could not send control message.\n");
         return -1;
      }

      if((retval=alsad_recv_control(sock, &rcvd_control)) <=0){
         fprintf(stderr,"alsad_reply_structs: Failed to recv control.\n");
         return -1;
      }
   
   }
   
   if((rcvd_control.ctrl & ALSAD_MSG_RQST_ERR) == ALSAD_MSG_RQST_ERR){ 
      fprintf(stderr,"alsad_reply_structs: Received error control message.\n");
      return -1; 
   } 
   else if((rcvd_control.ctrl & ALSAD_MSG_RQST_OK) == ALSAD_MSG_RQST_OK) 
      return 0;

   fprintf(stderr,"alsad_reply_structs: Invalid reply message.\n");
   return -1;
}  


void alsad_free_net_structs(alsad_net_structs_t *free_structs){


   if(free_structs->hw_params!=NULL)       free(free_structs->hw_params);
   if(free_structs->sw_params!=NULL)       free(free_structs->sw_params);
   if(free_structs->local_stream!=NULL)   free(free_structs->local_stream);
   if(free_structs->remote_stream!=NULL)  free(free_structs->remote_stream);
   if(free_structs->connect_addr!=NULL)    free(free_structs->connect_addr);
   if(free_structs->local_pipe!=NULL)      free(free_structs->local_pipe);
   if(free_structs->remote_pipe!=NULL)     free(free_structs->remote_pipe);

}

/** @}*/
