/*! \file alsad_data_pipe_list.c
 *  \brief This is the implementation of the Data-Pipe List Library
 *  \author Daniel R. Warren
 *  \version 1.0
 *  \date    November 2004
 *  \ingroup data_pipe_list
*/


 /** \defgroup data_pipe_list The Data-Pipe List Library
 *   \ingroup application_components
 *   \brief Library that provides a mechanism for storing data-pipe list nodes.
 * 
   
 */
 
/** \addtogroup data_pipe_list */
/** @{*/


#include "alsad_data_pipe_list.h"


int alsad_data_pipe_list_init(alsad_data_pipe_list_t *new_list,
                                               unsigned int num_data_pipes){
   int retval;
  
   if((retval=linked_list_init(&(new_list->pipe_list),num_data_pipes)) < 0){
      fprintf(stderr,"alsad_data_pipe_list_init: ");
      fprintf(stderr,"linked_list_init failed \n");
      return -1;
   }

   return 0;
}


/** \todo This function should look at the linked list and
 * destroy the lists only when they are empty
 */

void alsad_data_pipe_list_destroy(alsad_data_pipe_list_t *destroy_list){

   linked_list_destroy(destroy_list->pipe_list);

}



int alsad_data_pipe_node_init(alsad_data_pipe_node_t **new_data_pipe_node,
                                 alsad_data_pipe_t *new_data_pipe){
   

   alsad_connect_addr_t *stored_connect_addr;
   alsad_hw_params_t *stored_hw_params;
   alsad_sw_params_t *stored_sw_params;
   
   if(new_data_pipe==NULL){
      return -1;
   }
   
   if(new_data_pipe->connect_addr==NULL){
      return -1;
   }
   
   if(new_data_pipe->hw_params==NULL){
      return -1;
   }
    
   if(new_data_pipe->sw_params==NULL){
      return -1;
   }
    
   *new_data_pipe_node=(alsad_data_pipe_node_t *)malloc(sizeof(alsad_data_pipe_node_t));
   if(new_data_pipe_node==NULL){
      fprintf(stderr,"Could not allocate memory for new_data_pipe_node,");
      return -1; 
   }
   
   (*new_data_pipe_node)->closing=0;
   (*new_data_pipe_node)->thread_byte_id=0;
   
   (*new_data_pipe_node)->pipe=(alsad_data_pipe_t *)malloc(sizeof(alsad_data_pipe_t));
   if((*new_data_pipe_node)->pipe==NULL){
      fprintf(stderr,"Could not allocate memory for new_data_pipe_node->pipe,");
      free(new_data_pipe_node);
      return -1; 
   }
   
   (*new_data_pipe_node)->pipe->identifier=new_data_pipe->identifier;
      

   stored_connect_addr=(alsad_connect_addr_t *)malloc(sizeof(alsad_connect_addr_t));
   if(stored_connect_addr==NULL){
      fprintf(stderr,"Could not allocate memory for stored_connect_addr");
      free((*new_data_pipe_node)->pipe);
      free((*new_data_pipe_node));
      return -1; 
   }
   memcpy(stored_connect_addr, new_data_pipe->connect_addr, sizeof(alsad_connect_addr_t));
   (*new_data_pipe_node)->pipe->connect_addr=stored_connect_addr;


   stored_hw_params=(alsad_hw_params_t *)malloc(sizeof(alsad_hw_params_t));
   if(stored_hw_params==NULL){
      fprintf(stderr,"Could not allocate memory for stored_hw_params");
      free(stored_connect_addr);
      free((*new_data_pipe_node)->pipe);
      free((*new_data_pipe_node));
      return -1; 
   }
   memcpy(stored_hw_params, new_data_pipe->hw_params, sizeof(alsad_hw_params_t));
   (*new_data_pipe_node)->pipe->hw_params=stored_hw_params;


   stored_sw_params=(alsad_sw_params_t *)malloc(sizeof(alsad_sw_params_t));
   if(stored_sw_params==NULL){
      fprintf(stderr,"Could not allocate memory for stored_sw_params");
      free(stored_hw_params);
      free(stored_connect_addr);
      free((*new_data_pipe_node)->pipe);
      free((*new_data_pipe_node));
      return -1; 
   }
   memcpy(stored_sw_params, new_data_pipe->sw_params, sizeof(alsad_sw_params_t));
   (*new_data_pipe_node)->pipe->sw_params=stored_sw_params;

   return 0;
}





void alsad_data_pipe_node_destroy(alsad_data_pipe_node_t *destroy_pipe_node){

   free(destroy_pipe_node->pipe->sw_params);
   free(destroy_pipe_node->pipe->hw_params);
   free(destroy_pipe_node->pipe->connect_addr);
   free(destroy_pipe_node->pipe);
   free(destroy_pipe_node);

}




alsad_data_pipe_node_t *alsad_data_pipe_list_insert(
                                         alsad_data_pipe_list_t *list,
                                         alsad_data_pipe_node_t *new_node){
   int retval;
   
   if((retval=linked_list_insert(list->pipe_list,
                            alsad_data_pipe_node_compare_equal,
                            (void *)&(new_node->pipe->identifier),
                            (void *)new_node)) < 0){
      fprintf(stderr,"Could not insert data_pipe into linked list\n");
      return NULL;
   }

   return new_node;
   
}





alsad_data_pipe_node_t* alsad_data_pipe_list_remove(
                                            alsad_data_pipe_list_t *list,
                                            unsigned int ident){

   alsad_data_pipe_node_t *retptr;

   if((retptr=(alsad_data_pipe_node_t *)linked_list_remove(list->pipe_list,
                            alsad_data_pipe_node_compare_equal,
                            (void *)&ident)) == NULL){
      fprintf(stderr,"Could not remove data_pipe from linked list\n");
   }

   return retptr;


}



alsad_data_pipe_node_t *alsad_data_pipe_list_find(alsad_data_pipe_list_t *list,
            unsigned int ident){

   alsad_data_pipe_node_t *retptr;

   if((retptr=(alsad_data_pipe_node_t *)linked_list_search(list->pipe_list,
                            alsad_data_pipe_node_compare_equal,
                            (void *)&ident)) == NULL){
      fprintf(stderr,"Could not find data_pipe in linked list\n");
   }

   return retptr;

  
}



unsigned long alsad_data_pipe_find_slowest(alsad_data_pipe_list_t *list,
            unsigned long global_byte_id){

   alsad_data_pipe_node_t *retptr;

   if((retptr=(alsad_data_pipe_node_t *)linked_list_compare_all(list->pipe_list,
                            alsad_data_pipe_node_compare_greater_than,
                            (void *)&global_byte_id)) == NULL){
      return global_byte_id;
   }

   return retptr->thread_byte_id;

  
}


int alsad_data_pipe_list_send(int sock, alsad_data_pipe_list_t *list, int code){

   int retval;
   alsad_data_pipe_list_send_args_t list_args;
   
   if(list==NULL)
      return -1;
    
   list_args.code=code;
   list_args.sock=sock;
      
       
   if((retval=linked_list_process_all(list->pipe_list,
                                      alsad_data_pipe_node_send_pipe,
                                      (void *)&list_args)) < 0){ 
         fprintf(stderr,"Could not send all data_pipes\n");
         return -1;
   }

   
   return 0;

}


   
int alsad_data_pipe_node_send_pipe(void *data_pipe, void *args){

   alsad_control_t send_control;
   int retval;
   alsad_data_pipe_node_t *tmp_data_pipe_node=
                       (alsad_data_pipe_node_t*)data_pipe;
   alsad_data_pipe_list_send_args_t *tmp_args=
                       (alsad_data_pipe_list_send_args_t *)args;

   send_control.code=tmp_args->code;   
   send_control.ctrl=ALSAD_RECV_DATA_PIPE;
   if((retval=alsad_send_data_pipe_w_ctrl(
                             tmp_args->sock,
                             tmp_data_pipe_node->pipe,
                             &send_control)) <= 0){
      fprintf(stderr,"Could not send data pipe to client\n");
      return -1;
   }
   send_control.code=tmp_data_pipe_node->pipe->identifier;
   if(tmp_data_pipe_node->pipe->hw_params!=NULL){
      send_control.ctrl=ALSAD_RECV_HW_PARAMS;
      if((retval=alsad_send_hw_params_w_ctrl(
                       tmp_args->sock,
                       tmp_data_pipe_node->pipe->hw_params,
                                   &send_control)) <= 0){
         fprintf(stderr,"Could not send hw_params to client\n");
         return -1;
      }
   }   
   if(tmp_data_pipe_node->pipe->sw_params!=NULL){
      send_control.ctrl=ALSAD_RECV_SW_PARAMS;
      if((retval=alsad_send_sw_params_w_ctrl(
                       tmp_args->sock,
                       tmp_data_pipe_node->pipe->sw_params,
                                   &send_control)) <= 0){
         fprintf(stderr,"Could not send sw_params to client\n");
         return -1;
      }
   }   
   if(tmp_data_pipe_node->pipe->connect_addr!=NULL){
      send_control.ctrl=ALSAD_RECV_CONN_ADDR;
      if((retval=alsad_send_conn_addr_w_ctrl(
                         tmp_args->sock,
                         tmp_data_pipe_node->pipe->connect_addr,
                                   &send_control)) <= 0){
         fprintf(stderr,"Could not send conn_addr to client\n");
         return -1;
      }
   }   

   return 0;
}





int alsad_data_pipe_node_compare_equal(void *identifier, void *data_pipe_node){

   unsigned int *tmp_identifier;
   alsad_data_pipe_node_t *tmp_data_pipe_node;

   tmp_identifier=(unsigned int *)identifier; 
   tmp_data_pipe_node=(alsad_data_pipe_node_t *)data_pipe_node; 

   if((*tmp_identifier)==tmp_data_pipe_node->pipe->identifier)
      return 0;
    
   return -1;

}





int alsad_data_pipe_node_compare_greater_than(void *data_pipe_node_a,
                                              void *data_pipe_node_b,
                                              void *args){

   unsigned long *global_byte_id;
   alsad_data_pipe_node_t *tmp_data_pipe_node_a;
   alsad_data_pipe_node_t *tmp_data_pipe_node_b;
   
   tmp_data_pipe_node_a=(alsad_data_pipe_node_t *)data_pipe_node_a; 
   tmp_data_pipe_node_b=(alsad_data_pipe_node_t *)data_pipe_node_b; 
   global_byte_id=(unsigned long *)args;
   
   if(*global_byte_id - tmp_data_pipe_node_a->thread_byte_id > 
                        *global_byte_id - tmp_data_pipe_node_b->thread_byte_id)
      return 0;
    
   return -1;

}
       
/** @}*/
