/** \defgroup alsa_client The ALSA Client Application
 *   \ingroup the_applications
 *   \brief The client application used in conjuction with alsad
 **/
 
/** \file alsa_client.c
 *  \brief This is the main file for the alsa_client process
 *  \author Daniel R. Warren
 *  \version 1.0
 *  \date    November 2004
 *  \ingroup alsa_client
*/

 
 
/** \addtogroup alsa_client */
/** @{*/
  

#include <stdlib.h>
#include <stdio.h>
#include <sys/file.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/time.h>
#include <string.h>
#include <ctype.h>
#include <netdb.h>
#include <sys/time.h>
#include <pthread.h>
#include <linux/stddef.h>
#include <linux/un.h>
#include <getopt.h>


#include "log_lib.h"
#include "config_file.h"
#include "auth_lib.h"
#include "alsad_net_lib.h"
#include "socket_lib.h"
#include "alsad_shared_lib.h"


#define ALSA_CLIENT_PROC_NAME "alsa_client"
#define ALSA_CLIENT_LOG_FILE "alsa_client.log"

#define ALSA_CLIENT_CONFIG_FILE "/etc/alsad/alsa_client.conf"
#define ALSA_CLIENT_VERSION "0.0.1"

#define LOGMESSAGE(string, location) logmessage(string, location ,\
	     ALSA_CLIENT_PROC_NAME, ALSA_CLIENT_LOG_FILE, __FILE__, __LINE__,\
	     __FUNCTION__);



/* Function Prototypes */
//static void catch_signal(int signal);
void print_help();


int main(int argc, char **argv) {

   char *filename;  
   char *config_file;
   char *envptr;
   char *portptr;
   int channels_found;
   int retval;
   int handle;
   int verbosely;
   alsad_net_structs_t send_structs;
   alsad_config_keys_t load_keys;   
   alsad_control_t conn_ctrl;
   alsad_hw_params_t send_hw_params;
   alsad_sw_params_t send_sw_params;
   alsad_stream_props_t send_local_stream;
   alsad_stream_props_t send_remote_stream;
   alsad_data_pipe_t send_local_pipe;
   alsad_data_pipe_t send_remote_pipe;
   alsad_connect_addr_t conn_addr, remote_conn_addr;
   
   int ind_opts;
   int opt;
   char *optstring="c:C:d:D:f:F:hi:I:l:p:P:r:s:S:t:v:V:w:W:";
   static struct option longopts[] = {
	   {"help",0,0,'h'},
	   {"version",0,0,'v'},
	   {"verbose",0,0,'V'},
	   {"server",1,0,'s'},
	   {"location",1,0,'l'},
	   {"server_port",1,0,'p'},
	   {"command",1,0,'c'},
	   {"channels",1,0,'C'},
	   {"hw_iface",1,0,'w'},
	   {"data_pipe_id",1,0,'d'},
	   {"remote_data_pipe_id",1,0,'D'},
	   {"file",1,0,'f'},
	   {"rate",1,0,'r'},
	   {"stream_id",1,0,'i'},
	   {"remote_stream_id",1,0,'I'},
	   {"text_desc",1,0,'t'},
	   {"remote_text_desc",1,0,'T'},
	   {"remote_server",1,0,'S'},
	   {"remote_port",1,0,'P'},
       {"src_permission",1,0,'W'},
	   {0,0,0,0}
   };
   
   

   alsad_initialize_hw_params(&send_hw_params); 
   alsad_initialize_sw_params(&send_sw_params); 
   alsad_initialize_connect_addr(&conn_addr);    
   alsad_initialize_connect_addr(&remote_conn_addr); 
   alsad_initialize_stream_props(&send_local_stream); 
   alsad_initialize_stream_props(&send_remote_stream); 
   alsad_initialize_data_pipe(&send_remote_pipe); 
   alsad_initialize_data_pipe(&send_local_pipe); 
   
   
   // Defaults
   channels_found=0;
   filename=NULL;
   config_file=NULL;
   verbosely=0;
   conn_ctrl.ctrl=0;
   send_local_stream.text_desc[0]='\0';  
   send_local_pipe.identifier=ALSAD_DFLT_DP_IDENTIFIER;
   send_local_stream.rate=ALSAD_DFLT_RATE;
   send_local_stream.identifier=ALSAD_DFLT_IDENTIFIER;
   send_local_stream.src_permission=ALSAD_SRC_CALC_RATE |
                                                      ALSAD_SRC_SINK_RATE;
   

   envptr = getenv("ALSAD_SOUND");
   if (envptr != NULL){
      portptr=strchr(envptr,':');
      if(portptr!=NULL){
          *portptr='\0';
	  portptr++;
      }
      else{
         fprintf(stderr,"ALSAD_SOUND environment variable has no ':'\n");
	 exit(1);
      }
      strncpy(conn_addr.hostname,envptr,sizeof(conn_addr.hostname)); 
      conn_addr.port=atoi(portptr);
   }
   else{
      strncpy(conn_addr.hostname,ALSAD_DFLT_HOSTNAME,
		                           sizeof(conn_addr.hostname)); 
      conn_addr.port=atoi(ALSAD_DFLT_PORT);
   }
   
   
   
   

   while((opt = getopt_long(argc,argv, optstring, longopts, &ind_opts))!=-1){
      switch(opt){
      case 'c':
         if(!strcasecmp(optarg, "add_stream")){
            conn_ctrl.ctrl=ALSAD_RQST_ADD_STREAM;
         }else if(!strcasecmp(optarg, "del_stream")){
            conn_ctrl.ctrl=ALSAD_RQST_DEL_STREAM;
         }else if(!strcasecmp(optarg, "add_src")){
            conn_ctrl.ctrl=ALSAD_RQST_ADD_SRC;
         }else if(!strcasecmp(optarg,"del_src")){
            conn_ctrl.ctrl=ALSAD_RQST_DEL_SRC;
         }else if(!strcasecmp(optarg, "add_sink")){
            conn_ctrl.ctrl=ALSAD_RQST_ADD_SINK;
         }else if(!strcasecmp(optarg, "del_sink")){
            conn_ctrl.ctrl=ALSAD_RQST_DEL_SINK;
         }else if(!strcasecmp(optarg, "list_stream")){
            conn_ctrl.ctrl=ALSAD_RQST_LIST_STREAM;
         }else if(!strcasecmp(optarg, "list_pipe")){
            conn_ctrl.ctrl=ALSAD_RQST_LIST_PIPE;
         }else{
            fprintf(stderr,"controls: add_stream\n");
            fprintf(stderr,"          del_stream\n");
            fprintf(stderr,"          add_src\n");
            fprintf(stderr,"          del_src\n");
            fprintf(stderr,"          add_sink\n");
            fprintf(stderr,"          del_sink\n");
            fprintf(stderr,"          list_stream\n");
            fprintf(stderr,"          list_pipe\n");
            return 0;
         }
         break;
      case 'C':
         channels_found=1;
         send_local_stream.channels=atoi(optarg);
         break;
      case 'd':
         send_local_pipe.identifier=atoi(optarg);
         break;
      case 'D':
         send_remote_pipe.identifier=atoi(optarg);
         break;
      case 'f':
         filename = strdup(optarg);
         break;
      case 'F':
         config_file = strdup(optarg);
         break;
      case 'h':
         print_help(); 
         exit(0);
      case 'i':
         send_local_stream.identifier=atoi(optarg);
         break;
      case 'I':
         send_remote_stream.identifier=atoi(optarg);
         break;
      case 'l':
         if(!strcasecmp(optarg, "hw_iface")){
            conn_ctrl.ctrl=(conn_ctrl.ctrl | ALSAD_PATH_HW_IFACE);
         }else if(!strcasecmp(optarg, "current")){
            conn_ctrl.ctrl=(conn_ctrl.ctrl | ALSAD_PATH_CUR_SOCK);
         }else if(!strcasecmp(optarg, "remote")){
            conn_ctrl.ctrl=(conn_ctrl.ctrl | ALSAD_PATH_RMTE_SOCK);
         }else{
            fprintf(stderr,"locations: hw_iface\n");
            fprintf(stderr,"           current\n");
            fprintf(stderr,"           remote\n");
            return 0;
         }
         break;
      case 'p':
         conn_addr.port=atoi(optarg);
         break;
      case 'P':
         remote_conn_addr.port=atoi(optarg);
         break;
      case 'r':
         send_local_stream.rate=atoi(optarg);
         break;
      case 's':
         strncpy(conn_addr.hostname,optarg,ALSAD_MAX_HOST_LEN);
         break;
      case 'S':
         strcpy(remote_conn_addr.hostname, optarg);
         break;
      case 't':
         strcpy(send_local_stream.text_desc,optarg);
         break;
      //case 'T':
      //   strcpy(send_remote_stream.text_desc,optarg);
      //   break;
      case 'v':
         fprintf(stdout,"Version: alsad_client %s\n\n",ALSA_CLIENT_VERSION);
         exit(0);
      case 'V':
         verbosely=1;
         break;
      case 'w':
         strcpy(send_hw_params.pcm_name,optarg);
         break;
      case 'W':
         if(!strcasecmp(optarg, "calc_rate")){
            send_local_stream.src_permission=ALSAD_SRC_CALC_RATE;
         }else if(!strcasecmp(optarg, "sink_rate")){
            send_local_stream.src_permission=ALSAD_SRC_SINK_RATE;
         }else if(!strcasecmp(optarg, "calc_sink")){
            send_local_stream.src_permission=ALSAD_SRC_CALC_RATE |
                                               ALSAD_SRC_SINK_RATE;
         }else{
            fprintf(stderr,"source permission: calc_rate\n");
            fprintf(stderr,"                   sink_rate\n");
            fprintf(stderr,"                   calc_sink\n");
            return 0;
         }
         break;
      default:
         fprintf(stderr,"Invalid option exiting....\n");
         exit(1);
      };
   }




   
   //load parameters from the config file
   
   if(config_file==NULL){
      retval=alsad_config_key_load(ALSA_CLIENT_CONFIG_FILE,&load_keys);
      if(retval<0){
        fprintf(stderr,"Could not load default config file. Exiting\n");
        return 1;
      }
   }
   else{
      retval=alsad_config_key_load(config_file,&load_keys);
      if(retval<0){
        fprintf(stderr,"Could not load user defined config file. Exiting\n");
        return 1;
      }
   }

   //No command line argument for format   
   send_local_stream.format=snd_pcm_format_value(
                                        load_keys.config_pcm_format);
   if(send_local_stream.format==SND_PCM_FORMAT_UNKNOWN){
      fprintf(stderr,"PCM format given is unknown ....exiting\n\n");
      exit(1);
   }
   
   if(channels_found){
      send_local_stream.channels=atoi(load_keys.config_channels);
   }
   
   //No command line argument for circ_buff_size
   send_local_stream.circ_buff_size=atoi(load_keys.config_circ_buff_size);  
   
   //No command line arguemt for open_mode
   send_hw_params.open_mode=atoi(load_keys.config_open_mode);
   if(send_hw_params.open_mode!=0 && send_hw_params.open_mode!=1){
      send_hw_params.open_mode=atoi(ALSAD_DFLT_OPEN_MODE);
   }
   
   //No command line argument for buffer_time
   send_hw_params.buffer_time=atol(load_keys.config_card_buf_time);

   //No command line argument for period_time
   send_hw_params.period_time=atol(load_keys.config_card_period_time);
   
   //No command line argument for pcm_name
   memcpy(&send_hw_params.pcm_name,load_keys.config_card_device_name,
		                           ALSAD_MAX_DEV_LENGTH);
   
   //End loading parameters from file
   






   if(conn_ctrl.ctrl == 0){
      fprintf(stderr, "No control specified... Exiting\n");
      exit(1);
   }

   if((retval=strlen(send_local_stream.text_desc))==0 && filename!=NULL){
      strncpy(send_local_stream.text_desc,filename,sizeof(send_local_stream.text_desc));
   }

   if((conn_ctrl.ctrl & ALSAD_RQST_ADD_SINK) == ALSAD_RQST_ADD_SINK && 
      (conn_ctrl.ctrl & ALSAD_PATH_CUR_SOCK) == ALSAD_PATH_CUR_SOCK){
      remove(filename);
      if ((handle = open(filename, O_WRONLY | O_CREAT, 0644)) == -1) {
         fprintf(stderr,"Could not open file to capture file\n");
         return -1;
      }
   }
   else if((conn_ctrl.ctrl & ALSAD_RQST_ADD_SRC) == ALSAD_RQST_ADD_SRC && 
      (conn_ctrl.ctrl & ALSAD_PATH_CUR_SOCK) == ALSAD_PATH_CUR_SOCK){
      
      if ((handle = open(filename, O_RDONLY, 0)) == -1) {
         fprintf(stderr,"Could not open file to playback file\n");
         return -1;
      }
   }

   
   if((conn_ctrl.ctrl & ALSAD_PATH_RMTE_SOCK)!=ALSAD_PATH_RMTE_SOCK){
      if((retval=gethostname(remote_conn_addr.hostname,ALSAD_MAX_HOST_LEN)) < 0){
         fprintf(stderr,"Could not get local hostname\n");
	 return -1;
      }
      remote_conn_addr.port=0;
   }
   //Should these be set here or on the server.
   //   send_hw_params.stream=SND_PCM_STREAM_CAPTURE;
   //   send_hw_params.stream=SND_PCM_STREAM_PLAYBACK;

   if(verbosely){ 
      fprintf(stderr,"\n\n");

      fprintf(stderr,"Channels: %u\n",send_local_stream.channels);
      fprintf(stderr,"circ_buff_size: %u\n",send_local_stream.circ_buff_size);
      fprintf(stderr,"card_open_mode: %d\n",send_hw_params.open_mode);
      fprintf(stderr,"card_buffer_time: %d\n",send_hw_params.buffer_time);
      fprintf(stderr,"card_period_time: %d\n",send_hw_params.period_time);
      fprintf(stderr,"card_device_name: %s\n",send_hw_params.pcm_name);
      fprintf(stderr,"username: %s\n",load_keys.config_username);
      fprintf(stderr,"private_key: %s\n",load_keys.config_priv_key_file);

      fprintf(stderr,"\n\n");

   }

   //signal(SIGINT, catch_signal);
   //signal(SIGTERM, catch_signal);
   //signal(SIGABRT, catch_signal);

   send_structs.hw_params=&send_hw_params;
   send_structs.sw_params=&send_sw_params;
   send_structs.local_stream=&send_local_stream;
   send_structs.remote_stream=&send_remote_stream;
   send_structs.connect_addr=&remote_conn_addr;
   send_structs.local_pipe=&send_local_pipe;
   send_structs.remote_pipe=&send_remote_pipe;
   

   retval=alsad_conn_with_host(&conn_addr, &send_structs, 
		   load_keys.config_username,
		   load_keys.config_priv_key_file,
		   &conn_ctrl);

   if(retval<0){
      fprintf(stderr,"Connection with host failed\n");
      return -1;
   }
   
   retval=alsad_handle_client_protocol(retval,handle,&conn_ctrl);

   if(retval<0){
      fprintf(stderr,"Connection interupted\n");
      return -1;
   }
   fprintf(stderr,"Connection Terminated normally\n");
   return 0;
}  // End of main()


void print_help(){

   fprintf(stdout,"\nAvailable options:\n");
   fprintf(stdout,"-h help\n");
   fprintf(stdout,"-v version \n");
   fprintf(stdout,"-V verbose \n");
   fprintf(stdout,"-s <server> \n");
   fprintf(stdout,"-p <server_port> \n");
   fprintf(stdout,"-c <command> \n");
   fprintf(stdout,"-C <number_of_channels> \n");
   fprintf(stdout,"-w <hw_iface>\n");
   fprintf(stdout,"-d <data_pipe_id>\n");
   fprintf(stdout,"-D <remote_data_pipe_id>\n");
   fprintf(stdout,"-l <pipe_location>\n");
   fprintf(stdout,"-f <file> \n");
   fprintf(stdout,"-F <config_file> \n");
   fprintf(stdout,"-r <rate> \n");
   fprintf(stdout,"-i <stream_id> \n");
   fprintf(stdout,"-I <remote_stream_id>\n");
   fprintf(stdout,"-t <text_desc> \n");
   fprintf(stdout,"-W <src_permission>\n");
   fprintf(stdout,"-S <remote_server> \n");
   fprintf(stdout,"-P <remote_port> \n");

   fprintf(stdout,"\nValid commands:\n");
   fprintf(stdout,"\tadd_stream\n");
   fprintf(stdout,"\tdel_stream\n");
   fprintf(stdout,"\tadd_src\n");
   fprintf(stdout,"\tdel_src\n");
   fprintf(stdout,"\tadd_sink\n");
   fprintf(stdout,"\tdel_sink\n");
   fprintf(stdout,"\tlist_stream\n");
   fprintf(stdout,"\tlist_pipe\n\n");
   
   fprintf(stdout,"\nValid locations:\n");
   fprintf(stdout,"\thw_iface\n");
   fprintf(stdout,"\tcurrent\n");
   fprintf(stdout,"\tremote\n\n");

   fprintf(stdout,"\nValid source permissions:\n");
   fprintf(stdout,"\tcalc_rate\n");
   fprintf(stdout,"\tsink_rate\n");
   fprintf(stdout,"\tcalc_sink\n\n");

}


//static void catch_signal(int signal){
//   LOGMESSAGE("Exiting Carefully..... ", LOGSTDOUT);
//   exit(0);
//}


/** @}*/

